package com.zorkdata.apiservice.application.impl;

import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Lists;
import com.google.gson.Gson;
import com.zaxxer.hikari.HikariDataSource;
import com.zorkdata.apiservice.application.application.ApiApplication;
import com.zorkdata.apiservice.application.cache.DataSourceCache;
import com.zorkdata.apiservice.application.cache.UserCache;
import com.zorkdata.apiservice.application.dbutils.ConnectionManager;
import com.zorkdata.apiservice.application.dbutils.FreemarkUtil;
import com.zorkdata.apiservice.domain.assembler.ApiAssembler;
import com.zorkdata.apiservice.domain.assembler.ApiAuthAssembler;
import com.zorkdata.apiservice.domain.assembler.ApiGroupAssembler;
import com.zorkdata.apiservice.domain.domain.*;
import com.zorkdata.apiservice.domain.dto.*;
import com.zorkdata.apiservice.domain.listdto.WhiteListDTO;
import com.zorkdata.apiservice.domain.repository.ApiAuthRepository;
import com.zorkdata.apiservice.domain.repository.ApiCallRepository;
import com.zorkdata.apiservice.domain.repository.ApiGroupRepository;
import com.zorkdata.apiservice.domain.repository.ApiRepository;
import com.zorkdata.dddlib.core.sdk.InvokeResult;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.checkerframework.common.reflection.qual.Invoke;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import javax.persistence.criteria.Predicate;
import java.sql.*;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.*;

/**
 * @title: ApiApiServiceApplicationImpl
 * @Author Linxinfeng
 * @Date: 2021-09-2021/9/9 009-11:04
 */
@Component
@Slf4j
public class ApiServiceApplicationImpl implements ApiApplication {

    public static final String CREATE = "create";
    public static final String REGIST = "regist";
    public static final String FROM = "FROM";
    public static final String WHERE = "WHERE";
    public static final String SELECT = "SELECT";
    public static final String SELECT_TOP = "SELECT TOP ";
    public static final String SELECT_COUNT = "SELECT COUNT(*) ";
    public static final String AND = "AND";
    public static final String CONTENT = "content";
    public static final String PAGE_NUM = "pageNum";
    public static final String PAGE_SIZE = "pageSize";
    private static final String MYSQL_TYPE = "mysql";
    private static final String ELASTICSEARCH6_TYPE = "elasticsearch6";
    private static final String SQLSERVER_TYPE = "sqlserver";
    public static final String LIKE = "LIKE";
    public static final String NOT_LIKE = "NOT LIKE";
    public static final String IN = "IN";
    public static final String NOT_IN = "NOT IN";
    public static final String EQUAL = "=";
    public static final String UNEQUAL = "!=";
    public static final String GT = ">";
    public static final String GE = ">=";
    public static final String LT = "<";
    public static final String LE = "<=";
    public static final String POST = "POST";
    public static final String GET = "GET";
    public static final String PUT = "PUT";
    public static final String DELETE = "DELETE";
    public static final String HEAD = "head";
    public static final String QUERY = "query";
    public static final String BODY = "body";
    private final int NO_PAGE = 0;
    private final int PAGE_ABLE = 1;
    public static final String PATH_VARIABLE = "pathVariable";
    public static final String QUERY_VARIABLE = "queryVariable";
    private static final String NULL = "null";
    private static final Integer TWO = 2;
    private static final String SLASH = "/";
    private static final String ANDTWO = "&";


    @Autowired
    ApiGroupRepository apiGroupRepository;

    @Autowired
    ApiRepository apiRepository;

    @Autowired
    ApiAuthRepository apiAuthRepository;

    @Autowired
    ApiCallRepository apiCallRepository;

    @Autowired
    @Qualifier(value = "remoteRestTemplate")
    RestTemplate restTemplate;

    public static final String ORDER_ASC = "asc";

    @Value("${api-getway-address}")
    private String gateWayAddress;

    @Override
    public InvokeResult addGroup(ApiGroupDTO apiGroupDTO, String username) {
        InvokeResult invokeResult = null;
        try {
            Integer parentId = apiGroupDTO.getParentId();
            List<ApiGroup> apiGroups = apiGroupRepository.findByParentId(parentId);
            apiGroups.forEach(item -> {
                String name = item.getName();
                if (apiGroupDTO.getName().equals(name)) {
                    throw new RuntimeException("同级目录下名称不可重复");
                }
            });
            //id=0代表新增
            if (apiGroupDTO.getId().equals(0)) {
                apiGroupRepository.save(ApiGroupAssembler.toApiGroup(apiGroupDTO, username));
                invokeResult = InvokeResult.success("新增api分组成功");
            } else {
                apiGroupRepository.save(ApiGroupAssembler.toApiGroup(apiGroupDTO, username));
                invokeResult = InvokeResult.success("修改api分组成功");
            }
        } catch (Exception e) {
            invokeResult = InvokeResult.fail();
            invokeResult.setMessage("添加api分组失败!");
            log.error("添加api分组失败:{}", e.getMessage());
        }
        return invokeResult;
    }

    @Override
    public InvokeResult deleteGroup(Integer id) {
        ApiGroup apiGroup = new ApiGroup();
        InvokeResult invokeResult;
        apiGroup.setId(id);
        try {
            //删除接口分组,查询该id的分组是否存在parentId,如果没有可以删除,如果存在不能删除
            Integer byGroupId = apiGroupRepository.findByGroupId(id);
            Integer count = apiGroupRepository.selectParentId(id);
            if (count.equals(0) && byGroupId.equals(0)) {
                apiGroupRepository.delete(apiGroup);
                invokeResult = InvokeResult.success();
                invokeResult.setMessage("删除接口分组成功!");
            } else {
                log.debug("删除接口分组失败,所选删除分组可能含有子分组或者api!");
                invokeResult = InvokeResult.fail();
                invokeResult.setMessage("删除接口分组失败,所选删除分组可能含有子分组或者api!");
                return invokeResult;
            }
        } catch (Exception e) {
            log.error("删除接口分组失败,所选删除分组可能含有子分组或者api!", e);
            invokeResult = InvokeResult.fail();
            invokeResult.setMessage("删除接口分组失败,所选删除分组可能含有子分组或者api!" + e.getMessage());
        }
        return invokeResult;
    }

    @Override
    public InvokeResult getGroup(Integer id) {
        Specification<ApiGroup> specification = (root, criteriaQuery, criteriaBuilder) -> criteriaBuilder.equal(root.get("id"), id);
        InvokeResult invokeResult;
        try {
            //非空判断
            Integer count = apiGroupRepository.selectCountFromId(id);
            if (count > 0) {
                ApiGroup apiGroup = apiGroupRepository.findOne(specification).orElse(null);
                log.debug("查询单个接口分组成功!");
                invokeResult = InvokeResult.success();
                invokeResult.setData(apiGroup);
            } else {
                log.error("查询接口分组失败,id不存在!");
                invokeResult = InvokeResult.fail();
                invokeResult.setMessage("查询接口分组失败,id不存在!");
                return invokeResult;
            }
        } catch (Exception e) {
            log.error("查询单个接口分组失败!", e);
            invokeResult = InvokeResult.fail();
            invokeResult.setMessage("查询单个接口分组失败!" + e.getMessage());
        }
        return invokeResult;
    }

    @Override
    public Page<ApiGroup> listGroup(String keyword, Integer pageSize, Integer pageNo, String sortField, String sortType) {
        Pageable pageable = null;
        Specification<ApiGroup> specification = null;
        try {
            String sortFieldValue = StrUtil.isNotBlank(sortField) ? sortField : "createdTime";
            Sort.Direction sortDirection = Sort.Direction.DESC;
            if (ORDER_ASC.equalsIgnoreCase(sortType)) {
                sortDirection = Sort.Direction.ASC;
            }
            pageable = PageRequest.of((pageNo > 0 ? pageNo - 1 : pageNo), pageSize, Sort.by(sortDirection, sortFieldValue));
            specification = (root, criteriaQuery, criteriaBuilder) -> {
                List<Predicate> predicates = Lists.newArrayList();
                if (StrUtil.isNotBlank(keyword)) {
                    String kyword = StrUtil.builder("%", keyword, "%").toString();
                    Predicate keywordPredicate = criteriaBuilder.or(criteriaBuilder.like(root.get("name"), kyword));
                    predicates.add(keywordPredicate);
                }
                return criteriaBuilder.and(predicates.toArray(new Predicate[predicates.size()]));
            };
        } catch (Exception e) {
            log.error("列表查询api分组失败:{}", e.getMessage());
        }
        return apiGroupRepository.findAll(specification, pageable);
    }

    @Override
    public Api addApi(ApiDTO apiDTO, String username, Integer userid) {
        String name = apiDTO.getName();
        String querySql = apiDTO.getQuerySql();
        if (org.springframework.util.StringUtils.hasLength(querySql)) {
            apiDTO.setQuerySql(formatQuerySql(querySql));
        }
        Integer count = 0;
        try {
            count = apiRepository.selectApiName(name);
            //count=0代表名称不存在,可进行新增
            if (count.equals(0)) {
                apiDTO.setSourceWay("create");
                apiDTO.setPath("http://" + gateWayAddress + "/v1/dataService/api/executeApi/" + apiDTO.getName());
                apiRepository.save(ApiAssembler.toApi(apiDTO, username, userid));
            } else {
                if (apiDTO.getId().equals(0)) {
                    throw new RuntimeException("保存Api接口失败!,名称已存在");
                } else {
                    apiDTO.setSourceWay("create");
                    apiDTO.setPath("http://" + gateWayAddress + "/v1/dataService/api/executeApi/" + apiDTO.getName());
                    apiRepository.save(ApiAssembler.toApi(apiDTO, username, userid));
                }
            }
        } catch (Exception e) {
            throw new RuntimeException(e.getMessage());
        }
        return null;
    }


    @Override
    public Api registApi(ApiDTO apiDTO, String username, Integer userid) {
        String name = apiDTO.getName();
        Integer count = 0;
        try {
            count = apiRepository.selectApiName(name);
            //count=0代表名称不存在,可进行新增
            if (count.equals(0)) {
                apiDTO.setSourceWay("regist");
                apiDTO.setPath("http://" + gateWayAddress + "/v1/dataService/api/executeApi/" + apiDTO.getName());
                apiRepository.save(ApiAssembler.toApi(apiDTO, username, userid));
            } else {
                if (apiDTO.getId().equals(0)) {
                    throw new RuntimeException("保存Api接口失败!,名称已存在");
                } else {
                    apiDTO.setSourceWay("regist");
                    apiDTO.setPath("http://" + gateWayAddress + "/v1/dataService/api/executeApi/" + apiDTO.getName());
                    apiRepository.save(ApiAssembler.toApi(apiDTO, username, userid));
                }
            }
        } catch (Exception e) {
            throw new RuntimeException(e.getMessage());
        }
        return null;
    }

    @Override
    public InvokeResult rescindApi(Integer id) {
        Api api = new Api();
        InvokeResult invokeResult;
        api.setId(id);
        try {
            Integer status = apiGroupRepository.selectApiStatus(id);
            if (status.equals(1)) {
                apiGroupRepository.rescindApi(id);
                invokeResult = InvokeResult.success();
                invokeResult.setMessage("撤销发布Api接口成功!");
            } else {
                log.error("接口已经撤销,不能重复撤销!");
                invokeResult = InvokeResult.fail();
                invokeResult.setMessage("接口已经撤销,不能重复撤销!");
            }
        } catch (Exception e) {
            log.error("Api接口撤销发布失败!", e);
            invokeResult = InvokeResult.fail();
            invokeResult.setMessage("Api接口撤销发布失败" + e.getMessage());
        }
        return invokeResult;
    }

    @Override
    public InvokeResult deleteApi(Integer id) {
        Api api = new Api();
        InvokeResult invokeResult;
        api.setId(id);
        try {
            //删除接口分组,查询该id的分组是否存在parentId,如果没有可以删除,如果存在不能删除
            Integer count = apiRepository.selectCountFromId(id);
            if (count > 0) {
                apiRepository.delete(api);
            } else {
                log.error("删除Api接口失败!");
                invokeResult = InvokeResult.fail();
                invokeResult.setMessage("删除Api接口失败!");
                return invokeResult;
            }
            invokeResult = InvokeResult.success();
            invokeResult.setMessage("删除Api接口成功!");
        } catch (Exception e) {
            log.error("删除Api接口失败!", e);
            invokeResult = InvokeResult.fail();
            invokeResult.setMessage("删除Api接口失败" + e.getMessage());
        }
        return invokeResult;
    }

    @Override
    public ApiAuth addAuth(ApiAuthDTO apiAuthDTO, String username) {
        return apiAuthRepository.save(ApiAuthAssembler.toApiAuth(apiAuthDTO, username));
    }

    @Override
    public Integer selectUserId(ApiAuthDTO apiAuthDTO) {
        Integer userId = apiAuthDTO.getUserId();
        Integer apiId = apiAuthDTO.getApiId();
        return apiAuthRepository.selectUserId(userId, apiId);
    }

    @Override
    public Integer selectApiStatus(ApiAuthDTO apiAuthDTO) {
        Integer apiId = apiAuthDTO.getApiId();
        return apiAuthRepository.selectApiStatus(apiId);
    }

    @Override
    public InvokeResult deleteAuth(Integer id) {
        ApiAuth apiAuth = new ApiAuth();
        InvokeResult invokeResult;
        apiAuth.setId(id);
        try {
            /**
             * 删除接口分组
             *查询该id的分组是否存在parentId,如果没有可以删除,如果存在不能删除
             *判断用户是否已经授权
             */
            Integer id1 = apiAuthRepository.selectApiId(id);
            Integer status = apiAuthRepository.selectApiStatus(id1);
            if (status.equals(1)) {
                //status 为 1 说明api已经发布,接口授权可以删除
                apiAuthRepository.delete(apiAuth);
                invokeResult = InvokeResult.success();
                invokeResult.setMessage("删除Api接口成功!");
            } else {
                log.error("Api接口未发布,不可删除授权!");
                invokeResult = InvokeResult.fail();
                invokeResult.setMessage("Api接口未发布,不可删除授权!");
                return invokeResult;
            }
        } catch (Exception e) {
            log.error("删除Api接口失败!", e);
            invokeResult = InvokeResult.fail();
            invokeResult.setMessage("删除Api接口失败" + e.getMessage());
        }
        return invokeResult;
    }

    @Override
    public InvokeResult publishApi(Integer id) {
        Api api = new Api();
        InvokeResult invokeResult;
        api.setId(id);
        try {
            Integer status = apiGroupRepository.selectApiStatus(id);
            if (status.equals(0)) {
                //0表示未发布
                apiGroupRepository.publishApi(id);
                invokeResult = InvokeResult.success();
                invokeResult.setMessage("发布Api接口成功!");
            } else {
                log.error("接口已经发布,不能重复发布!");
                invokeResult = InvokeResult.fail();
                invokeResult.setMessage("接口已经发布,不能重复发布!");
            }
        } catch (Exception e) {
            log.error("Api接口发布失败!", e);
            invokeResult = InvokeResult.fail();
            invokeResult.setMessage("Api接口发布失败" + e.getMessage());
        }
        return invokeResult;
    }


    @Override
    public InvokeResult getMyApi(Integer userId, String groupId, String keyword, Integer pageNo, Integer pageSize, String sortField, String sortType, String sourceWay, String datasource) {
        InvokeResult invokeResult;
        try {
            //非空判断,存在userid
            Integer count = apiRepository.selectCountFromUserId(userId);
            if (count != 0) {
                String sortFieldValue = StrUtil.isNotBlank(sortField) ? sortField : "createdTime";
                Sort.Direction sortDirection = Sort.Direction.DESC;
                if (ORDER_ASC.equalsIgnoreCase(sortType)) {
                    sortDirection = Sort.Direction.ASC;
                }
                Pageable pageable = PageRequest.of((pageNo > 0 ? pageNo - 1 : pageNo), pageSize, Sort.by(sortDirection, sortFieldValue));
                Specification<Api> specification = (root, criteriaQuery, criteriaBuilder) -> {
                    List<Predicate> predicates = Lists.newArrayList();
                    if (StrUtil.isNotBlank(keyword)) {
                        String kyword = StrUtil.builder("%", keyword, "%").toString();
                        Predicate keywordPredicate1 = criteriaBuilder.and(criteriaBuilder.like(root.get("name"), kyword));
                        predicates.add(keywordPredicate1);
                    }
                    if (StringUtils.isNotBlank(datasource)) {
                        String datasourceWord = StrUtil.builder("%", datasource, "%").toString();
                        Predicate keywordPredicate2 = criteriaBuilder.and(criteriaBuilder.like(root.get("datasource"), datasourceWord));
                        predicates.add(keywordPredicate2);
                    }
                    if (StringUtils.isNotBlank(sourceWay)) {
                        Predicate keywordPredicate3 = criteriaBuilder.and(criteriaBuilder.equal(root.get("sourceWay"), sourceWay));
                        predicates.add(keywordPredicate3);
                    }
                    Optional.ofNullable(userId).ifPresent(val -> {
                        predicates.add(criteriaBuilder.equal(root.get("userId"), val));
                    });
                    Optional.ofNullable(groupId).ifPresent(val -> {
                        predicates.add(criteriaBuilder.equal(root.get("group").get("id"), val));
                    });
                    return criteriaBuilder.and(predicates.toArray(new Predicate[predicates.size()]));
                };
                //Page获取全部返回结果
                Page<Api> all = apiRepository.findAll(specification, pageable);
                //处理返回结果
                List<Api> content = all.getContent();
                long totalElements = all.getTotalElements();
                List<ApiViewDTO> apiViewDTOList = new ArrayList<>();
                content.forEach(item -> {
                    ApiViewDTO apiViewDTO = new ApiViewDTO();
                    apiViewDTO.setId(item.getId());
                    apiViewDTO.setName(item.getName());
                    apiViewDTO.setPath("http://" + gateWayAddress + "/v1/dataService/api/executeApi/" + item.getName());
                    apiViewDTO.setDescription(item.getDescription());
                    apiViewDTO.setApiAuthDTOList(ApiAuthAssembler.toApiAuthDTOList(item.getApiAuthList()));
                    String datasourceStr = item.getDatasource();
                    DataSource dataSource = JSONObject.parseObject(datasourceStr, DataSource.class);
                    if (null == dataSource) {
                        apiViewDTO.setDateSource("{}");
                    } else {
                        apiViewDTO.setDateSource(dataSource.getName());
                    }
                    apiViewDTO.setCreatedBy(item.getCreatedBy());
                    //24小时调用量
                    Integer apiId = item.getId();
                    Date endTime = new Date();
                    long time = endTime.getTime();
                    long day = 24 * 60 * 60 * 1000;
                    Date startTime = new Date(time - day);

                    Integer count24 = apiRepository.findLastCall24ByApiCall(apiId, startTime, endTime);
                    apiViewDTO.setLastCall24(count24);
                    //调用量
                    Integer countByApiCall = apiRepository.findCountByApiCall(apiId);
                    apiViewDTO.setTotalCall(countByApiCall);
                    apiViewDTO.setSouceWay(item.getSourceWay());
                    apiViewDTO.setCreatedTime(item.getCreatedTime());
                    apiViewDTO.setCreatedBy(item.getCreatedBy());
                    apiViewDTO.setStatus(item.getStatus());
                    apiViewDTOList.add(apiViewDTO);
                });
                QueryResultDTO queryResult = new QueryResultDTO();
                queryResult.setPageNum(pageNo);
                queryResult.setPageSize(pageSize);
                queryResult.setTotalNum((int) totalElements);
                queryResult.setData(apiViewDTOList);
                log.debug("查看我创建的Api成功!");
                invokeResult = InvokeResult.success();
                invokeResult.setData(queryResult);
            } else {
                log.debug("查看我创建的Api失败,userId不存在!");
                invokeResult = InvokeResult.fail();
                invokeResult.setMessage("查询接口分组失败,userId不存在!");
                return invokeResult;
            }
        } catch (Exception e) {
            log.error("查看我创建的Api失败!", e);
            invokeResult = InvokeResult.fail();
            invokeResult.setMessage("查看我创建的Api失败!" + e.getMessage());
        }
        return invokeResult;
    }

    @Override
    public InvokeResult listMarketApi(String keyword, Integer pageNo, Integer pageSize, String sortField, String sortType, Long startTime, Long endTime) {
        InvokeResult invokeResult;
        Pageable pageable = null;
        Specification<Api> specification = null;
        try {
            String sortFieldValue = StrUtil.isNotBlank(sortField) ? sortField : "updatedTime";
            Sort.Direction sortDirection = Sort.Direction.DESC;
            if (ORDER_ASC.equalsIgnoreCase(sortType)) {
                sortDirection = Sort.Direction.ASC;
            }
            pageable = PageRequest.of((pageNo > 0 ? pageNo - 1 : pageNo), pageSize, Sort.by(sortDirection, sortFieldValue));
            specification = (root, criteriaQuery, criteriaBuilder) -> {
                List<Predicate> predicates = Lists.newArrayList();
                if (StrUtil.isNotBlank(keyword)) {
                    String kyword = StrUtil.builder("%", keyword, "%").toString();
                    Predicate keywordPredicate = criteriaBuilder.or(criteriaBuilder.like(root.get("name"), kyword), criteriaBuilder.like(root.get("description"), kyword), criteriaBuilder.like(root.get("createdBy"), kyword));
                    predicates.add(keywordPredicate);
                }
                if (ObjectUtils.isNotEmpty(startTime) && ObjectUtils.isNotEmpty(endTime)) {
                    if (!startTime.equals(0) && !endTime.equals(0)) {
                        try {
                            DateTime date1 = DateUtil.date(startTime);
                            DateTime date2 = DateUtil.date(endTime);
                            SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd");
                            String format = simpleDateFormat.format(date1);
                            String format2 = simpleDateFormat.format(date2);
                            Date start = simpleDateFormat.parse(format);
                            Date end = simpleDateFormat.parse(format2);
                            Predicate keywordPredicate = criteriaBuilder.and(criteriaBuilder.between(root.get("updatedTime"), start, end));
                            predicates.add(keywordPredicate);
                        } catch (ParseException e) {
                            log.error("时间转换失败" + e.getMessage());
                            return null;
                        }
                    }
                }
                return criteriaBuilder.and(predicates.toArray(new Predicate[predicates.size()]));
            };
        } catch (Exception e) {
            log.error("列表查询市场api失败:{}", e.getMessage());
        }
        Page<Api> all = apiRepository.findAll(specification, pageable);
        List<Api> content = all.getContent();
        long totalElements = all.getTotalElements();
        List<ApiDTO> apiDTO = new ArrayList<>();
        content.forEach(item -> {
            ApiDTO apiDtO = new ApiDTO();
            String str = item.getRequestParam();
            List<RequestParamDTO> requestParamDTOList = JSON.parseArray(str, RequestParamDTO.class);
            apiDtO.setId(item.getId());
            apiDtO.setName(item.getName());
            apiDtO.setDescription(item.getDescription());
            apiDtO.setCreatedBy(item.getCreatedBy());
            Integer apiId = apiDtO.getId();
            Integer countByApiCall = apiRepository.findCountByApiCall(apiId);
            apiDtO.setTotalCall(countByApiCall);
            apiDtO.setCreatedBy(item.getCreatedBy());
            apiDtO.setUpdatedTime(item.getUpdatedTime());
            apiDtO.setRequestType(item.getRequestType());
            apiDtO.setReturnType(item.getReturnType());
            apiDtO.setRequestParam(requestParamDTOList);
            apiDtO.setPage(item.getPage());
            apiDTO.add(apiDtO);
        });
        QueryResultDTO queryResult = new QueryResultDTO();
        queryResult.setPageNum(pageNo);
        queryResult.setPageSize(pageSize);
        queryResult.setTotalNum((int) totalElements);
        queryResult.setData(apiDTO);
        invokeResult = InvokeResult.success(queryResult, "查询市场api成功!");
        return invokeResult;
    }


    @Override
    public InvokeResult listMyAuth(String username, Integer pageNo, Integer pageSize, String sortField, String sortType, Long startTime, Long endTime, String keyword, Integer userid) {
        InvokeResult invokeResult = null;
        List<Api> apis = null;
        try {
            if (null == keyword) {
                keyword = "";
            }
            if (null != startTime && null != endTime) {
                //修改时间Long转String
                DateTime date1 = DateUtil.date(startTime);
                DateTime date2 = DateUtil.date(endTime);
                SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd");
                String start = simpleDateFormat.format(date1);
                String end = simpleDateFormat.format(date2);
                Integer count = apiRepository.listMyAuthCount(keyword, userid, start, end);
                apis = apiRepository.listMyAuth(keyword, sortType, sortField, pageNo - 1, pageSize, userid, start, end);
                List<ApiDTO> apiDTO = new ArrayList<>();
                apis.forEach(item -> {
                    ApiDTO apiDtO = new ApiDTO();
                    String str = item.getRequestParam();
                    List<RequestParamDTO> requestParamDTOList = JSON.parseArray(str, RequestParamDTO.class);
                    apiDtO.setId(item.getId());
                    apiDtO.setName(item.getName());
                    apiDtO.setDescription(item.getDescription());
                    apiDtO.setCreatedBy(item.getCreatedBy());
                    Integer apiId = apiDtO.getId();
                    Integer countByApiCall = apiRepository.findCountByApiCall(apiId);
                    apiDtO.setTotalCall(countByApiCall);
                    apiDtO.setCreatedBy(item.getCreatedBy());
                    apiDtO.setUpdatedTime(item.getUpdatedTime());
                    apiDtO.setRequestType(item.getRequestType());
                    apiDtO.setReturnType(item.getReturnType());
                    apiDtO.setRequestParam(requestParamDTOList);
                    apiDtO.setPage(item.getPage());
                    apiDTO.add(apiDtO);
                });
                QueryResultDTO queryResult = new QueryResultDTO();
                queryResult.setPageNum(pageNo);
                queryResult.setPageSize(pageSize);
                queryResult.setTotalNum(count);
                queryResult.setData(apiDTO);
                invokeResult = InvokeResult.success(queryResult, "查询市场api成功!");
                return invokeResult;
            } else {
                Integer count = apiRepository.listMyAuthTwoCount(keyword, userid);
                apis = apiRepository.listMyAuthTwo(keyword, sortType, sortField, pageNo - 1, pageSize, userid);
                List<ApiDTO> apiDTO = new ArrayList<>();
                apis.forEach(item -> {
                    ApiDTO apiDtO = new ApiDTO();
                    String str = item.getRequestParam();
                    List<RequestParamDTO> requestParamDTOList = JSON.parseArray(str, RequestParamDTO.class);
                    apiDtO.setId(item.getId());
                    apiDtO.setName(item.getName());
                    apiDtO.setDescription(item.getDescription());
                    apiDtO.setCreatedBy(item.getCreatedBy());
                    Integer apiId = apiDtO.getId();
                    Integer countByApiCall = apiRepository.findCountByApiCall(apiId);
                    apiDtO.setTotalCall(countByApiCall);
                    apiDtO.setCreatedBy(item.getCreatedBy());
                    apiDtO.setUpdatedTime(item.getUpdatedTime());
                    apiDtO.setRequestType(item.getRequestType());
                    apiDtO.setReturnType(item.getReturnType());
                    apiDtO.setRequestParam(requestParamDTOList);
                    apiDtO.setPage(item.getPage());
                    apiDTO.add(apiDtO);
                });
                QueryResultDTO queryResult = new QueryResultDTO();
                queryResult.setPageNum(pageNo);
                queryResult.setPageSize(pageSize);
                queryResult.setTotalNum(count);
                queryResult.setData(apiDTO);
                invokeResult = InvokeResult.success(queryResult, "查询市场api成功!");
                return invokeResult;
            }
        } catch (Exception e) {
            log.error("查看我授权的Api失败" + e.getMessage());
            invokeResult = InvokeResult.fail();
            invokeResult.setMessage("查看我授权的Api失败!" + e.getMessage());
            return invokeResult;
        }
    }


    @Override
    public InvokeResult dataView(Integer userId, Integer apiId, String username) {
        Boolean flag = false;
        List<Integer> authUserids = apiAuthRepository.findByApiId(apiId);
        //根据apiId获取Auth的createdBy,也就是授权者name
        if (authUserids.contains(userId)) {
            flag = true;
            return InvokeResult.success(flag);
        }
        //获取白名单,判断是否是白名单
        String users = apiRepository.findWhiteList(apiId);
        List<WhiteListDTO> whiteListDTO = JSON.parseArray(users, WhiteListDTO.class);
        if (whiteListDTO.size() != 0) {
            for (int i = 0; i < whiteListDTO.size(); i++) {
                WhiteListDTO user = whiteListDTO.get(i);
                Integer id = user.getId();
                //  判断是否是白名单
                if (userId.equals(id)) {
                    flag = true;
                    return InvokeResult.success(flag);
                }
            }
        }
        //判断是否是创建者
        Integer createUid = apiRepository.findCreateUidByApiId(apiId);
        if (createUid.equals(userId)) {
            flag = true;
        }
        return InvokeResult.success(flag);
    }


    @Override
    public InvokeResult listAllUser() {
        List<User> users = UserCache.getUsers();
        InvokeResult invokeResult = InvokeResult.success();
        invokeResult.setData(users);
        return invokeResult;
    }

    @Override
    public InvokeResult queryDatasourceList() {
        Map<String, DataSource> dataSources = DataSourceCache.getDataSources();
        InvokeResult invokeResult = InvokeResult.success();
        invokeResult.setData(dataSources);
        return invokeResult;
    }

    @Override
    public InvokeResult queryApiSingle(Integer id) {
        Specification<Api> specification = (root, criteriaQuery, criteriaBuilder) -> criteriaBuilder.equal(root.get("id"), id);
        InvokeResult invokeResult;
        try {
            Api api = apiRepository.findOne(specification).orElse(null);
            ApiDTO apiDTO = ApiAssembler.toApiDTO(api);
            invokeResult = InvokeResult.success();
            invokeResult.setData(apiDTO);
        } catch (Exception e) {
            log.error("异常了,", e);
            invokeResult = InvokeResult.fail();
            invokeResult.setMessage("查询失败" + e.getMessage());
        }
        return invokeResult;
    }

    @Override
    public InvokeResult queryAuthSingle(Integer apiId) {
        Specification<ApiAuth> specification = (root, criteriaQuery, criteriaBuilder) -> criteriaBuilder.equal(root.get("apiId"), apiId);
        InvokeResult invokeResult;
        try {
            List<ApiAuth> apiAuth = apiAuthRepository.findAll(specification);
            invokeResult = InvokeResult.success();
            invokeResult.setData(apiAuth);
        } catch (Exception e) {
            log.error("异常了,", e);
            invokeResult = InvokeResult.fail();
            invokeResult.setMessage("查询失败" + e.getMessage());
        }
        return invokeResult;
    }

    @Override
    public InvokeResult getApiLimit() {
        InvokeResult invokeResult;
        try {
            List<Map<String, Integer>> limitAndName = apiRepository.findLimitAndName();
            invokeResult = InvokeResult.success();
            invokeResult.setData(limitAndName);
        } catch (Exception e) {
            log.error("异常了,", e);
            invokeResult = InvokeResult.fail();
            invokeResult.setMessage("查询失败" + e.getMessage());
        }
        return invokeResult;
    }

    @Override
    public InvokeResult executeApi(String apiName, Map<String, String> inParam, String username, Integer userid) {
        //开始时间
        Long startTime = System.currentTimeMillis();
        InvokeResult invokeResult = null;
        Api api = apiRepository.findByName(apiName);
        //获取api的id,根据apiId去查询该id的Api给哪些用户授权了,对其进行限制
        Integer apiId = api.getId();
        String createdBy = api.getCreatedBy();
        //1.判断是不是创建者,如果是创建者,直接跳过,可以直接调用,不受权限控制
        if (!username.equals(createdBy)) {
            Map<Date, Date> startTimeAndendTime = apiAuthRepository.findApiAuthByApiId(apiId, userid);
            Date start = startTimeAndendTime.get("start_time");
            Date end = startTimeAndendTime.get("end_time");
            //2.判断有无设置权限,size=0说明根本没有设置权限,可以调用
            if (startTimeAndendTime.size() > 0) {
                //3.判断开始结束时间是否为null,如果为null,说明选择了对时间不进行限制操作
                if (null != start && null != end) {
                    Date nowDate = new Date();
                    int i1 = nowDate.compareTo(start);
                    int i2 = end.compareTo(nowDate);
                    //4.比较时间,符合 start > nowdate > end,不符合的话直接返回,权限不足
                    if (i1 != 1 || i2 != 1) {
                        invokeResult = InvokeResult.fail();
                        invokeResult.setMessage("权限不足!");
                        return invokeResult;
                    }
                }
            }
        }
        //0未发布 , 1发布
        Integer status = apiRepository.findStatusByName(apiName);
        if (1 != status) {
            invokeResult = InvokeResult.fail();
            invokeResult.setMessage("名称为[" + apiName + "]的接口未发布!");
            return invokeResult;
        }
        if (null == api) {
            invokeResult = InvokeResult.fail();
            invokeResult.setMessage("名称为[" + apiName + "]的接口不存在!");
            return invokeResult;
        }

        String sourceWay = api.getSourceWay();
        if (CREATE.equals(sourceWay)) {
            String datasourceStr = api.getDatasource();
            DataSourceDTO dataSource = JSONObject.parseObject(datasourceStr, DataSourceDTO.class);
            String dataSourceType = dataSource.getDatasourceType();
            switch (dataSourceType) {
                case MYSQL_TYPE:
                    invokeResult = executeMySqlApi(api, dataSource, inParam);
                    break;
                case SQLSERVER_TYPE:
                    invokeResult = executeSqlServerApi(api, dataSource, inParam);
                    break;
                default:
                    break;
            }
        } else {
            invokeResult = executeCommonApi(api, inParam);
        }
        Long endTime = System.currentTimeMillis();
        Long duration = endTime - startTime;
        ApiCall apiCall = new ApiCall();
        apiCall.setApiId(api.getId());
        apiCall.setCallTime(new DateTime(startTime));
        apiCall.setDuration(duration);
        String resquest = new Gson().toJson(inParam);
        apiCall.setRequestParam(resquest);
        String result = new Gson().toJson(invokeResult);
        apiCall.setResult(result);
        apiCall.setCaller(username);
        if (invokeResult.getCode().equals(0)) {
            apiCall.setSuccess(true);
        } else {
            apiCall.setSuccess(false);
        }
        apiCallRepository.save(apiCall);

        return invokeResult;
    }


    private InvokeResult executeMySqlApi(Api api, DataSourceDTO dataSource, Map<String, String> inParam) {
        InvokeResult invokeResult = null;
        String apiName = api.getName();
        String querySql = api.getQuerySql();
        String countSql = null;
        String dataTable = api.getDataTable();
        Integer page = api.getPage();
        PageParamDTO pageParamDTO = new PageParamDTO();
        boolean pageAble = PAGE_ABLE == page ? true : false;

        String returnParamStr = api.getReturnParam();
        JSONArray returnParamJsonArray = JSONArray.parseArray(returnParamStr);
        List<ReturnParam> returnParams = returnParamJsonArray.toJavaList(ReturnParam.class);
        String requestParamStr = api.getRequestParam();
        JSONArray requestParamJsonArray = JSONArray.parseArray(requestParamStr);
        List<RequestParam> requestParams = requestParamJsonArray.toJavaList(RequestParam.class);
        if (!checkParams(requestParams, inParam, pageAble, pageParamDTO)) {
            invokeResult = InvokeResult.fail();
            invokeResult.setMessage("[" + apiName + "]的接口参数校验不通过!");
            return invokeResult;
        }

        if (StringUtils.isEmpty(querySql)) {
            StringBuilder sbCount = new StringBuilder(SELECT_COUNT);
            StringBuilder sbSql = new StringBuilder(SELECT + " ");
            for (ReturnParam returnParam : returnParams) {
                String bindField = returnParam.getBindField();
                sbSql.append(bindField).append(",");
            }
            sbSql.deleteCharAt(sbSql.length() - 1);
            sbSql.append(" ").append(FROM).append(" ").append(dataTable);
            sbCount.append(FROM).append(" ").append(dataTable);
            if (requestParams.size() > 0) {
                sbSql.append(" ").append(WHERE).append(" ");
                sbCount.append(" ").append(WHERE).append(" ");
            }
            for (int i = 0; i < requestParams.size(); i++) {
                RequestParam requestParam = requestParams.get(i);
                String bindField = requestParam.getBindField();
                String operator = requestParam.getOperator();
                String name = requestParam.getName();
                String value = inParam.get(name);
                if (null == value) {
                    continue;
                }
                if (i == 0) {
                    switch (operator) {
                        case LIKE:
                        case NOT_LIKE:
                            sbSql.append(bindField).append(" ").append(operator).append(" ").append("'%").append(value).append("%'");
                            sbCount.append(bindField).append(" ").append(operator).append(" ").append("'%").append(value).append("%'");
                            break;
                        case IN:
                        case NOT_IN:
                            sbSql.append(bindField).append(" ").append(operator).append(" ").append("(\"").append(value).append("\")");
                            sbCount.append(bindField).append(" ").append(operator).append(" ").append("(\"").append(value).append("\")");
                            break;
                        case EQUAL:
                        case UNEQUAL:
                        case GT:
                        case GE:
                        case LT:
                        case LE:
                            sbSql.append(bindField).append(" ").append(operator).append(" ").append("\"").append(value).append("\"");
                            sbCount.append(bindField).append(" ").append(operator).append(" ").append("\"").append(value).append("\"");
                            break;
                        default:
                            break;
                    }
                } else {
                    sbSql.append(AND).append(" ").append(bindField).append(" ").append(operator).append(" ").append(value);
                    sbCount.append(AND).append(" ").append(bindField).append(" ").append(operator).append(" ").append(value);
                }
            }
            if (!pageAble) {
                sbSql.append(" limit 2000");
            } else {
                int from = (pageParamDTO.getPageNum() - 1) * pageParamDTO.getPageSize();
                sbSql.append(" limit ").append(from).append(",").append(pageParamDTO.getPageSize());
            }
            countSql = sbCount.toString();
            querySql = sbSql.toString();
        } else {
            Map<String, Object> params = new HashMap<>(requestParams.size());
            for (RequestParam requestParam : requestParams) {
                String name = requestParam.getName();
                String value = inParam.get(name);
                if (StringUtils.isEmpty(value)) {
                    value = requestParam.getDefaultValue();
                }
                params.put(name, value);
            }

            querySql = FreemarkUtil.freemarkerProcess(params, querySql, CONTENT);
            String[] split = querySql.split(FROM);

            if (pageAble) {
                countSql = "SELECT COUNT(*) FROM " + split[1];
                int from = (pageParamDTO.getPageNum() - 1) * pageParamDTO.getPageSize();
                querySql += " limit " + from + "," + pageParamDTO.getPageSize();
            } else {
                querySql += " limit 2000";
            }
        }
        HikariDataSource hikariDataSource = ConnectionManager.getDataSorce(dataSource.getName());
        if (null != hikariDataSource) {
            try {
                Connection connection = hikariDataSource.getConnection();
                if (pageAble) {
                    invokeResult = mySqlPageQuery(countSql, querySql, connection, pageParamDTO);
                } else {
                    invokeResult = sqlNoPageQuery(querySql, connection);
                }
            } catch (SQLException e) {
                log.error("获取数据库连接失败:{}", e.getMessage());
                invokeResult = InvokeResult.fail();
                invokeResult.setMessage("获取数据库连接失败:" + e.getMessage());
            }
        } else {
            invokeResult = InvokeResult.fail();
            invokeResult.setMessage("数据源连接池不存在");
        }
        return invokeResult;
    }

    private InvokeResult executeSqlServerApi(Api api, DataSourceDTO dataSource, Map<String, String> inParam) {
        InvokeResult invokeResult = null;
        Integer page = api.getPage();
        boolean pageAble = PAGE_ABLE == page ? true : false;
        if (pageAble) {
            invokeResult = InvokeResult.fail();
            invokeResult.setMessage("SQL server 不支持分页功能!");
            return invokeResult;
        }

        String apiName = api.getName();
        String querySql = api.getQuerySql();
        String dataTable = api.getDataTable();

        String returnParamStr = api.getReturnParam();
        JSONArray returnParamJsonArray = JSONArray.parseArray(returnParamStr);
        List<ReturnParam> returnParams = returnParamJsonArray.toJavaList(ReturnParam.class);
        String requestParamStr = api.getRequestParam();
        JSONArray requestParamJsonArray = JSONArray.parseArray(requestParamStr);
        List<RequestParam> requestParams = requestParamJsonArray.toJavaList(RequestParam.class);
        if (!checkParams(requestParams, inParam, pageAble, null)) {
            invokeResult = InvokeResult.fail();
            invokeResult.setMessage("[" + apiName + "]的接口参数校验不通过!");
            return invokeResult;
        }

        Map<String, String> selectFields = new HashMap<>(returnParams.size());
        if (StringUtils.isEmpty(querySql)) {
            StringBuilder sbSql = new StringBuilder(SELECT_TOP);
            sbSql.append(2000).append(" ");
            for (ReturnParam returnParam : returnParams) {
                String bindField = returnParam.getBindField();
                String type = returnParam.getType();
                sbSql.append(bindField).append(",");
                selectFields.put(bindField, type);
            }
            sbSql.deleteCharAt(sbSql.length() - 1);
            sbSql.append(FROM).append(" ").append(dataTable);
            if (requestParams.size() > 0) {
                sbSql.append(" ").append(WHERE).append(" ");
            }
            for (int i = 0; i < requestParams.size(); i++) {
                RequestParam requestParam = requestParams.get(i);
                String bindField = requestParam.getBindField();
                String operator = requestParam.getOperator();
                String name = requestParam.getName();
                String value = inParam.get(name);
                if (null == value) {
                    continue;
                }
                if (i == 0) {
                    switch (operator) {
                        case LIKE:
                        case NOT_LIKE:
                            sbSql.append(bindField).append(" ").append(operator).append(" ").append("'%").append(value).append("%'");
                            break;
                        case IN:
                        case NOT_IN:
                            sbSql.append(bindField).append(" ").append(operator).append(" ").append("(\"").append(value).append("\")");
                            break;
                        case EQUAL:
                        case UNEQUAL:
                        case GT:
                        case GE:
                        case LT:
                        case LE:
                            sbSql.append(bindField).append(" ").append(operator).append(" ").append("\"").append(value).append("\"");
                            break;
                        default:
                            break;
                    }
                } else {
                    sbSql.append(AND).append(" ").append(bindField).append(bindField).append(" ").append(operator).append(" ").append(value);
                }
            }
            querySql = sbSql.toString();
        } else {
            Map<String, Object> params = new HashMap<>(requestParams.size());
            for (RequestParam requestParam : requestParams) {
                String name = requestParam.getName();
                String value = inParam.get(name);
                if (StringUtils.isEmpty(value)) {
                    value = requestParam.getDefaultValue();
                }
                params.put(name, value);
            }

        }
        HikariDataSource hikariDataSource = ConnectionManager.getDataSorce(dataSource.getName());
        if (null != hikariDataSource) {
            try {
                Connection connection = hikariDataSource.getConnection();
                invokeResult = sqlNoPageQuery(querySql, connection);
            } catch (SQLException e) {
                log.error("获取数据库连接失败:{}", e.getMessage());
                invokeResult = InvokeResult.fail();
                invokeResult.setMessage("获取数据库连接失败:" + e.getMessage());
            }
        } else {
            invokeResult = InvokeResult.fail();
            invokeResult.setMessage("数据源连接池不存在");
        }
        return invokeResult;
    }

    private boolean checkParams(List<RequestParam> requestParams, Map<String, String> inParam, boolean pageAble, PageParamDTO pageParamDTO) {
        try {
            if (pageAble) {
                String pageNum = inParam.get(PAGE_NUM);
                if (StringUtils.isEmpty(pageNum)) {
                    log.error("分页参数:pageNum为空!");
                    return false;
                } else {
                    int pagNo = Integer.parseInt(pageNum);
                    if (pagNo == 0) {
                        log.error("分页参数:pageNum必需大于零!");
                        return false;
                    }
                    pageParamDTO.setPageNum(pagNo);
                }

                String pageSize = inParam.get(PAGE_SIZE);
                if (StringUtils.isEmpty(pageSize)) {
                    log.error("分页参数:pageSize为空!");
                    return false;
                } else {
                    int pagsize = Integer.parseInt(pageSize);
                    if (pagsize == 0) {
                        log.error("分页参数:pageSize必需大于零!");
                        return false;
                    }
                    pageParamDTO.setPageSize(pagsize);
                }
            }
            for (RequestParam requestParam : requestParams) {
                String name = requestParam.getName();
                String value = inParam.get(name);
                if (null == value) {
                    Integer requried = requestParam.getRequried();
                    if (requried > 0) {
                        log.error("接口参数:{}不可以为空", name);
                        return false;
                    }
                }
            }
        } catch (Exception e) {
            log.error("接口参数校验不通过");
            return false;
        }
        return true;
    }

    private boolean checkParams(List<RequestParam> requestParams, List<ConstParam> constParams, Map<String, String> inParam) {
        try {
            for (RequestParam requestParam : requestParams) {
                String name = requestParam.getName();
                String value = inParam.get(name);
                if (null == value) {
                    Integer requried = requestParam.getRequried();
                    if (requried > 0) {
                        log.error("接口参数:{}不可以为空", name);
                        return false;
                    }
                }
            }

            for (ConstParam constParam : constParams) {
                String name = constParam.getName();
                String value = inParam.get(name);
                if (null == value) {
                    String defaultValue = constParam.getDefaultValue();
                    if (StringUtils.isEmpty(defaultValue)) {
                        log.error("接口常量参数:{}不可以为空", name);
                        return false;
                    }
                }
            }
        } catch (Exception e) {
            log.error("接口参数校验不通过");
            return false;
        }
        return true;
    }

    private InvokeResult mySqlPageQuery(String countSql, String querySql, Connection connection, PageParamDTO pageParamDTO) {
        InvokeResult invokeResult = null;
        Statement statement = null;
        ResultSet resultSet = null;
        List<Map<String, String>> dataResult = null;
        QueryResultDTO queryResult = null;
        try {
            statement = connection.createStatement();
            resultSet = statement.executeQuery(countSql);
            int totalNum = 0;
            while (resultSet.next()) {
                totalNum = resultSet.getInt(1);
            }
            if (totalNum > 0) {
                resultSet = statement.executeQuery(querySql);
                ResultSetMetaData data = resultSet.getMetaData();
                dataResult = new ArrayList<Map<String, String>>();
                while (resultSet.next()) {
                    Map<String, String> map = new HashMap<String, String>(16);
                    for (int i = 1; i <= data.getColumnCount(); i++) {
                        String columnName = data.getColumnName(i);
                        String value = resultSet.getString(columnName);
                        map.put(columnName, value);
                    }
                    dataResult.add(map);
                }
                queryResult = new QueryResultDTO(totalNum, pageParamDTO.getPageNum(), pageParamDTO.getPageSize(), dataResult);
                invokeResult = InvokeResult.success(queryResult, "接口调用成功!");
            } else {
                invokeResult = InvokeResult.success(null, "未查询到符合条件的数据");
            }
        } catch (SQLException e) {
            log.error("获取数据库连接失败:{}", e.getMessage());
        } finally {
            if (resultSet != null) {
                try {
                    resultSet.close();
                } catch (Exception e) {
                    log.error("数据集关闭失败:{}", e.getMessage());
                }
            }
            if (statement != null) {
                try {
                    statement.close();
                } catch (Exception e) {
                    log.error("statement关闭失败:{}", e.getMessage());
                }
            }
        }
        return invokeResult;
    }

    private InvokeResult sqlNoPageQuery(String querySql, Connection connection) {
        InvokeResult invokeResult = null;
        Statement statement = null;
        ResultSet resultSet = null;
        List<Map<String, String>> dataResult = null;
        try {
            statement = connection.createStatement();
            resultSet = statement.executeQuery(querySql);
            ResultSetMetaData data = resultSet.getMetaData();
            dataResult = new ArrayList<Map<String, String>>();
            while (resultSet.next()) {
                Map<String, String> map = new HashMap<String, String>(16);
                for (int i = 1; i <= data.getColumnCount(); i++) {
                    String columnName = data.getColumnName(i);
                    String value = resultSet.getString(columnName);
                    map.put(columnName, value);
                }
                dataResult.add(map);
            }
            invokeResult = InvokeResult.success(dataResult, "接口调用成功");
        } catch (SQLException e) {
            log.error("获取数据库连接失败:{}", e.getMessage());
        } finally {
            if (resultSet != null) {
                try {
                    resultSet.close();
                } catch (Exception e) {
                    log.error("数据集关闭失败:{}", e.getMessage());
                }
            }
            if (statement != null) {
                try {
                    statement.close();
                } catch (Exception e) {
                    log.error("statement关闭失败:{}", e.getMessage());
                }
            }
        }
        return invokeResult;
    }


    private InvokeResult executeCommonApi(Api api, Map<String, String> inParam) {
        InvokeResult invokeResult = null;
        String apiName = api.getName();

        List<RequestParam> requestParams = null;
        String requestParamStr = api.getRequestParam();
        if (!NULL.equals(requestParamStr)) {
            JSONArray requestParamJsonArray = JSONArray.parseArray(requestParamStr);
            requestParams = requestParamJsonArray.toJavaList(RequestParam.class);
        }

        List<ConstParam> constParams = null;
        String constParamStr = api.getConstParams();
        if (!NULL.equals(constParamStr)) {
            JSONArray constParamJsonArray = JSONArray.parseArray(constParamStr);
            constParams = constParamJsonArray.toJavaList(ConstParam.class);
        }

        if (null != inParam && null != requestParams && null != constParams) {
            if (!checkParams(requestParams, constParams, inParam)) {
                invokeResult = InvokeResult.fail();
                invokeResult.setMessage("[" + apiName + "]的接口参数校验不通过!");
                return invokeResult;
            }
        }


        HttpHeaders httpHeaders = extractHttpHeaders(requestParams, constParams, inParam);
        String queryParams = "";
        Map<String, String> bodyParams = new HashMap<>(16);

        queryParams = extractQueryParams(requestParams, constParams, inParam);
        bodyParams = extractBodyParams(requestParams, constParams, inParam);


        String sourceHost = api.getSourceHost();
        String sourcePath = api.getSourcePath();
        String requestType = api.getRequestType();
        String protocol = api.getProtocol();
        if (PUT.equals(requestType)) {
            if (bodyParams.size() == 0) {
                bodyParams.put("arg", "arg");
            }
        }

        String bodyStrParams = JSON.toJSONString(bodyParams);
        HttpEntity<String> httpEntity = null;


        String uir = protocol + "://" + sourceHost + sourcePath;
        if (queryParams.length() > TWO) {
            if (sourcePath.endsWith(SLASH)) {
                if (queryParams.startsWith(SLASH)) {
                    uir += queryParams.substring(1);
                } else {
                    uir += queryParams;
                }
            } else {
                if (queryParams.startsWith(SLASH)) {
                    uir += queryParams;
                } else {
                    uir += "/" + queryParams;
                }
            }
        }


        if (bodyParams.size() > 0) {
            httpEntity = new HttpEntity<String>(bodyStrParams, httpHeaders);
        } else {
            httpEntity = new HttpEntity<String>(httpHeaders);
        }

        String result = null;
        try {
            switch (requestType) {
                case GET:
                    result = restTemplate.getForObject(uir, String.class, httpEntity);
                    invokeResult = InvokeResult.success(result, "接口GET请求成功");
                    break;
                case POST:
                    result = restTemplate.postForObject(uir, httpEntity, String.class);
                    invokeResult = InvokeResult.success(result, "接口POST请求成功");
                    break;
                case PUT:
                    restTemplate.put(uir, httpEntity);
                    invokeResult = InvokeResult.success("{'message':'接口PUT请求成功!'}", "接口PUT请求成功!");
                    break;
                case DELETE:
                    restTemplate.delete(uir, httpEntity);
                    invokeResult = InvokeResult.success("{'message':'接口DELETE请求成功!'}", "接口DELETE请求成功!");
                    break;
                default:
                    invokeResult = InvokeResult.fail();
                    invokeResult.setMessage("请求类型:" + requestType + "不支持!");
                    break;
            }

        } catch (Exception e) {
            invokeResult = InvokeResult.fail();
            invokeResult.setMessage("接口请求错误:!" + e.getMessage());
        }
        return invokeResult;
    }

    private HttpHeaders extractHttpHeaders
            (List<RequestParam> requestParams, List<ConstParam> constParams, Map<String, String> inParam) {
        HttpHeaders httpHeaders = new HttpHeaders();
        httpHeaders.setContentType(MediaType.APPLICATION_JSON);

        if (null != requestParams && null != constParams) {
            for (RequestParam requestParam : requestParams) {
                if (HEAD.equals(requestParam.getPosition())) {
                    Integer requried = requestParam.getRequried();
                    if (requried > 0) {
                        httpHeaders.add(requestParam.getName(), inParam.get(requestParam.getName()));
                    } else {
                        httpHeaders.add(requestParam.getName(), requestParam.getDefaultValue());
                    }
                }
            }
            for (ConstParam constParam : constParams) {
                if (HEAD.equals(constParam.getPosition())) {
                    httpHeaders.add(constParam.getName(), constParam.getDefaultValue());
                }
            }
        }
        return httpHeaders;
    }

    private String extractQueryParams
            (List<RequestParam> requestParams, List<ConstParam> constParams, Map<String, String> inParam) {
        StringBuilder stringBuilder = new StringBuilder();
        List<RequestParam> pathRequsetVariables = new ArrayList<>();
        List<RequestParam> queryRequsetVariables = new ArrayList<>();
        if (null != requestParams) {
            for (RequestParam requestParam : requestParams) {
                if (QUERY.equals(requestParam.getPosition())) {
                    if (PATH_VARIABLE.equals(requestParam.getType())) {
                        pathRequsetVariables.add(requestParam);
                    } else if (QUERY_VARIABLE.equals(requestParam.getType())) {
                        queryRequsetVariables.add(requestParam);
                    }
                }
            }
        }
        List<ConstParam> pathConstVariables = new ArrayList<>();
        List<ConstParam> queryConstVariables = new ArrayList<>();
        if (null != constParams) {
            for (ConstParam constParam : constParams) {
                if (QUERY.equals(constParam.getPosition())) {
                    if (PATH_VARIABLE.equals(constParam.getType())) {
                        pathConstVariables.add(constParam);
                    } else if (QUERY_VARIABLE.equals(constParam.getType())) {
                        queryConstVariables.add(constParam);
                    }
                }
            }
        }
        if (pathRequsetVariables.size() > 0 || pathConstVariables.size() > 0) {
            for (RequestParam requestParam : pathRequsetVariables) {
                Integer requried = requestParam.getRequried();
                if (requried > 0) {
                    stringBuilder.append("/").append(inParam.get(requestParam.getName()));
                } else {
                    stringBuilder.append("/").append(requestParam.getDefaultValue());
                }
            }

            for (ConstParam constParam : pathConstVariables) {
                stringBuilder.append("/").append(constParam.getDefaultValue());
            }
        }
        if (queryRequsetVariables.size() > 0 || queryConstVariables.size() > 0) {
            stringBuilder.append("?");
            for (RequestParam queryRequsetVariable : queryRequsetVariables) {
                Integer requried = queryRequsetVariable.getRequried();
                if (requried > 0) {
                    stringBuilder.append(queryRequsetVariable.getName()).append("=").append(inParam.get(queryRequsetVariable.getName()));
                } else {
                    stringBuilder.append(queryRequsetVariable.getName()).append("=").append(queryRequsetVariable.getDefaultValue());
                }
                stringBuilder.append("&");
            }
            for (ConstParam queryConstVariable : queryConstVariables) {
                stringBuilder.append(queryConstVariable.getName()).append("=").append(queryConstVariable.getDefaultValue());
                stringBuilder.append("&");
            }
        }
        String result = stringBuilder.toString();
        if (result.endsWith(ANDTWO)) {
            result = result.substring(0, result.length() - 1);
        }
        return result;
    }

    private Map<String, String> extractBodyParams
            (List<RequestParam> requestParams, List<ConstParam> constParams, Map<String, String> inParam) {
        Map<String, String> bodyParams = new HashMap<>(16);
        if (null != requestParams) {
            for (RequestParam requestParam : requestParams) {
                if (BODY.equals(requestParam.getPosition())) {
                    Integer requried = requestParam.getRequried();
                    if (requried > 0) {
                        bodyParams.put(requestParam.getName(), inParam.get(requestParam.getName()));
                    } else {
                        bodyParams.put(requestParam.getName(), requestParam.getDefaultValue());
                    }
                }
            }
        }
        if (null != constParams) {
            for (ConstParam constParam : constParams) {
                if (BODY.equals(constParam.getPosition())) {
                    bodyParams.put(constParam.getName(), constParam.getDefaultValue());
                }
            }
        }
        return bodyParams;
    }

    private String formatQuerySql(String querySql) {
        return querySql.replace("$ { ", "${").replace(" }", "}");
    }
}
