package com.zork.other2jrsjcs.component;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.zork.other2jrsjcs.dao.JrsjcsTypeaBizDao;
import com.zork.other2jrsjcs.model.JrsjcsTypeaBizDO;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.concurrent.BasicThreadFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.http.*;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import static com.zork.common.utils.DateUtil.*;
import static com.zork.other2jrsjcs.constant.OtherConstant.*;


/**
 * @Author: Prock.Liy
 * @Date： 2022/4/12  5.26机器
 * @Description： 从ES查询other索引数据写入MySql
 */
@Slf4j
@Component
public class OtherQueryComponent implements ApplicationRunner {

    @Value("${size}")
    private String size;

    @Value("${initial}")
    private String initial;

    @Value("${initial-day}")
    private String initialDay;

    /**
     * esUrl
     */
    @Value("${esUrl}")
    private String esUrl;

    /**
     * 查询间隔分钟
     */
    @Value("${interval-ms}")
    private Integer intervalMs;

    /**
     * 查询频率，分钟
     */
    @Value("${initialDelay}")
    private Integer initialDelay;

    @Resource
    private JrsjcsTypeaBizDao jrsjcsTypeaBizDao;

    @Resource
    private RestTemplate restTemplate;

    /**
     * thread pools   如果线程池长度超过处理需要，可灵活回收空闲线程，若无可回收，则新建线程
     */
    protected ExecutorService EXECUTOR_POOL = Executors.newCachedThreadPool();

    protected ScheduledExecutorService EXECUTOR_SERVICE = Executors.newScheduledThreadPool(1,
            new BasicThreadFactory.Builder()
                    .namingPattern("EXAMPLE-SCHEDULE-POOL-%d")
                    .daemon(true).build());

    /**
     * 根据intervalMs条件查询es数据写入Mysql
     */
    @Override
    public void run(ApplicationArguments args) {
        try {
            log.info("esUrl: {}", String.format(esUrl, yearMonthDayPoint()));
            EXECUTOR_SERVICE.scheduleAtFixedRate(() -> {
                try {
                    Map<String, Object> map = getOther(false);
                    List<JrsjcsTypeaBizDO> jrsjcsTypeaBizDOList = (List<JrsjcsTypeaBizDO>) map.get(JRSJCS_TYPEA_BIZDO_LIST);
                    // 无数据直接退出
                    if (jrsjcsTypeaBizDOList.size() == 0) {
                        return;
                    }
                    jrsjcsTypeaBizDao.saveAll(jrsjcsTypeaBizDOList);
                } catch (Exception e) {
                    log.info("scheduleAtFixedRate Exception: {}", e.getMessage());
                }
            }, initialDelay, intervalMs, TimeUnit.MINUTES);
            log.info("Start Initialize query all day....");
            Map<String, Object> map = getOther(true);
            List<JrsjcsTypeaBizDO> jrsjcsTypeaBizDOList = (List<JrsjcsTypeaBizDO>) map.get(JRSJCS_TYPEA_BIZDO_LIST);
            // 无数据直接退出
            if (jrsjcsTypeaBizDOList.size() == 0) {
                return;
            }
            jrsjcsTypeaBizDao.saveAll(jrsjcsTypeaBizDOList);

            log.info("Start interval {}Minute query....", intervalMs);
        } catch (Exception e) {
            log.error("\n Query OtherDataExecute: -> {}", e.getMessage());
        }
    }

    /**
     * 从Es中查询数据,查询最近指定几分钟内数据
     *
     * @return List<JrsjcsTypeaBizDO>
     */
    private Map<String, Object> getOther(boolean findDay) {
        Map<String, Object> map = new HashMap<>(2);

        List<JrsjcsTypeaBizDO> jrsjcsTypeaBizDOList = new ArrayList<>();
        JSONObject dataResult = new JSONObject();

        String startTime = format8Gage(lastFewMinutes(intervalMs));
        String endTime = format8Gage(new Date());

        // 拼接es查询条件，根据size、logTypeName，最近几分钟内数据;true为初始化需要查询全天数据,false查询间隔时间段内数据
        String param = StringUtils.EMPTY;
        if (findDay) {
            param = String.format(initialDay, size);
        } else {
            param = String.format(initial, size, startTime, endTime);
        }

        try {
            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.APPLICATION_JSON);
            HttpEntity entity = new HttpEntity<>(JSONObject.toJSON(param), headers);
            ResponseEntity<JSONObject> result = restTemplate.exchange(String.format(esUrl, yearMonthDayPoint()), HttpMethod.POST, entity, JSONObject.class);

            dataResult = result.getBody();
            map.put(SCROLL_ID, dataResult.getString(SCROLL_ID));
            int dataSize = dataResult.getJSONObject(HITS).getJSONArray(HITS).size();

            // 判断是否有数据返回
            if (dataSize > 0) {
                JSONArray jsonArray = dataResult.getJSONObject(HITS).getJSONArray(HITS);
                for (Object o : jsonArray) {
                    JSONObject jsonObject = JSONObject.parseObject(JSON.toJSONString(o));
                    JSONObject normalFields = jsonObject.getJSONObject(SOURCE).getJSONObject(NORMAL_FIELDS);
                    // 转换UTC时间格式为时间戳
                    long timestamp = parseSSSTime(jsonObject.getJSONObject(SOURCE).getString(TIME_STAMP));
                    jrsjcsTypeaBizDOList.add(JrsjcsTypeaBizDO.builder()
                            .dst(normalFields.getString(DST))
                            .src(normalFields.getString(SCR))
                            .message(normalFields.getString(MESSAGE))
                            .type(normalFields.getString(TYPE))
                            .status(normalFields.getString(STATUS))
                            .time(yearMonthDayPoint())
                            .TIMESTAMP(timestamp)
                            .build()
                    );
                }
                map.put(JRSJCS_TYPEA_BIZDO_LIST, jrsjcsTypeaBizDOList);
                log.info("getOtherData SUCCESS,DataSize: -> {}... startTime:{},endTime:{}", dataSize, startTime, endTime);
            } else {
                map.put(JRSJCS_TYPEA_BIZDO_LIST, Collections.emptyList());
                log.info("getOtherData SUCCESS,No Data.. startTime:{},endTime:{}", startTime, endTime);
            }
        } catch (Exception e) {
            log.error("getOtherData Exception, query param:{}  \n Exception: -> {} ", param, e.getMessage());
        }
        return map;
    }

}
