package com.zork.zhtt.component;

import com.alibaba.fastjson.JSONObject;
import com.zork.zhtt.constant.DataSourceConstants;
import com.zork.zhtt.dto.SchedulesDTO;
import com.zork.zhtt.dto.SystemDTO;
import com.zork.zhtt.properties.SchedulesProperties;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.*;
import org.springframework.jdbc.core.BeanPropertyRowMapper;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.scheduling.annotation.SchedulingConfigurer;
import org.springframework.scheduling.config.ScheduledTaskRegistrar;
import org.springframework.scheduling.support.CronTrigger;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import java.text.SimpleDateFormat;
import java.util.*;

import static com.zork.common.constant.DateConstant.STANDARD_DATE_FORMAT_UTC;
import static com.zork.zhtt.constant.LogDataConstant.*;

/**
 * @Author: Prock.Liy
 * @Date： 2021/6/1
 * @Description： 数据统计组件
 */
@Slf4j
@Component
public class DataCountComponent implements SchedulingConfigurer {

    /**
     * cron表达式初始化
     */
    @Resource
    private SchedulesProperties schedulesProperties;

    @Resource
    @Qualifier("jdbcTemplateQudkf")
    private JdbcTemplate jdbcTemplateQudkf;

//    @Resource
//    @Qualifier("jdbcTemplateWskh")
//    private JdbcTemplate jdbcTemplateWskh;

    @Resource
    @Qualifier("jdbcTemplateWskhSlave1")
    private JdbcTemplate jdbcTemplateWskhSlave1;

    @Value("${spring.datasource.qudkf.url}")
    private String qudkfUrl;

//    @Value("${spring.datasource.wskh.url}")
//    private String wskhUrl;

    @Value("${spring.datasource.wskhslave1.url}")
    private String wskhslave1Url;

    @Value("${logstash.post-url}")
    private String logstashUrl;

    @Resource
    private RestTemplate restTemplate;

    /**
     * 初始化定时任务,cron表达式来自schedules.yml配置文件
     *
     * @param scheduledTaskRegistrar scheduledTaskRegistrar
     */
    @Override
    public void configureTasks(ScheduledTaskRegistrar scheduledTaskRegistrar) {
        schedulesProperties.getSchedulesDTOList().forEach(schedulesDTO -> {
            scheduledTaskRegistrar.addTriggerTask(
                    // 调用业务方法
                    () -> execute(schedulesDTO),
                    //2.设置执行周期(Trigger)
                    triggerContext -> {
                        //2.3 返回执行周期(Date)
                        return new CronTrigger(schedulesDTO.getCron()).nextExecutionTime(triggerContext);
                    }
            );
        });
    }

    /**
     * 执行Sql输出统计Json
     *
     * @param schedulesDTO schedulesDTO
     */
    public void execute(SchedulesDTO schedulesDTO) {
        try {
            Integer countResult = 0;
            String ip = StringUtils.EMPTY;
            String appSystem = StringUtils.EMPTY;
            Map<String, Integer> measures = new HashMap<>();
            switch (schedulesDTO.getDatasource()) {
                case DataSourceConstants.QUDKF:
                    ip = StringUtils.substringBefore(StringUtils.substringAfter(qudkfUrl, "@"), ":");
                    appSystem = DataSourceConstants.QUDKF_SYSTEM;
                    countResult = jdbcTemplateQudkf.queryForObject(schedulesDTO.getSql(), Integer.class);
                    measures.put(COUNT, countResult);
                    break;
//                case DataSourceConstants.WSKH:
//                    ip = StringUtils.substringBefore(StringUtils.substringAfter(wskhUrl, "@"), ":");
//                    appSystem = DataSourceConstants.WSKH_SYSTEM;
//                    countResult = jdbcTemplateWskh.queryForObject(schedulesDTO.getSql(), Integer.class);
//                    measures.put(COUNT, countResult);
//                    break;
                case DataSourceConstants.WSKH_SLAVE1:
                    ip = StringUtils.substringBefore(StringUtils.substringAfter(wskhslave1Url, "@"), ":");
                    appSystem = DataSourceConstants.WSKH_SYSTEM;

                    List<SystemDTO> systemDTOList = jdbcTemplateWskhSlave1.query(schedulesDTO.getSql(), new BeanPropertyRowMapper<>(SystemDTO.class));
                    if (systemDTOList.isEmpty()) {
                        log.info("本次查询无数据! 数据库ip地址: {} ,sql: {}", ip, schedulesDTO.getSql());
                        return;
                    }

                    measures = buildMeasuresValue(systemDTOList, schedulesDTO.getMetricSetName());
                    break;
            }
            dataConversionJson(schedulesDTO.getMetricSetName(), measures, appSystem, ip);
        } catch (Exception e) {
            log.error("\n Execute执行异常: -> {}\n" +
                    "执行SQL: ->{}", e.getMessage(), schedulesDTO.getSql());
        }
    }

    /**
     * @param systemDTOList
     * @param metricSetName
     * @return
     */
    private Map<String, Integer> buildMeasuresValue(List<SystemDTO> systemDTOList, String metricSetName) {
        Map<String, Integer> measures = new HashMap<>(systemDTOList.size());
        for (SystemDTO systemDTO : systemDTOList) {
            String key = StringUtils.EMPTY;
            // 获取measures中映射简称
            if (metricSetName.equals(LOG_KHYW)) {
                key = dataMapping(systemDTO.getFunc_no());
            } else {
                key = dataMapping(systemDTO.getOperate_name());
            }
            if (StringUtils.isNotEmpty(key)) {
                measures.put(key, Objects.isNull(systemDTO.getCount()) ? 0 : systemDTO.getCount());
            }
        }
        return measures;
    }

    /**
     * Json格式转换
     *
     * @param metricSetName metricSetName
     * @param measures      measures
     * @param ip            数据库ip
     * @return JSONObject
     */
    private void dataConversionJson(String metricSetName, Map<String, Integer> measures, String appSystem, String ip) {
        try {
            JSONObject jsonObject = new JSONObject();
            jsonObject.put("metricsetname", metricSetName);
            jsonObject.put("type", "jdbc");

            // before 8 hour
            Calendar cal = Calendar.getInstance();
            cal.setTime(new Date());
            cal.add(Calendar.HOUR_OF_DAY, -8);
            jsonObject.put("timestamp", new SimpleDateFormat(STANDARD_DATE_FORMAT_UTC).format(cal.getTime()));

            Map<String, String> dimensions = new HashMap<>(2);
            dimensions.put("appsystem", appSystem);
            dimensions.put("hostname", ip);
            jsonObject.put("dimensions", dimensions);

            jsonObject.put("measures", measures);

//            System.out.println("结果:" + jsonObject);

            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.APPLICATION_JSON);
            HttpEntity entity = new HttpEntity<>(jsonObject, headers);
            ResponseEntity<String> result = restTemplate.exchange(logstashUrl, HttpMethod.POST, entity, String.class);

            // 调用失败缺少measure值时，进行补齐重新调归补用递
            if (!result.getBody().contains(OK)) {
                String missing = StringUtils.substringBeforeLast(StringUtils.substringAfterLast(result.getBody(), MEASURE), IS);
                measures.put(missing, 0);
                dataConversionJson(metricSetName, measures, appSystem, ip);
            }
            log.info("转换JSON格式后: ->{}  ,Logstash接口调用Result: ->{}", jsonObject, result.getBody());
        } catch (Exception e) {
            e.printStackTrace();
            log.error("\n Sql转换异常: -> {} \t\t\t\t\t。参数:metricSetName:{}, measures:{} ,appSystem:{},ip:{}",
                    e.getMessage(), metricSetName, measures, appSystem, ip);
        }
    }

    /**
     * 获取数据库查询后对应简称
     *
     * @param key
     * @return
     */
    private static String dataMapping(String key) {
        JSONObject dataMapping = JSONObject.parseObject(DATA_BASE_MAPPING);
        return dataMapping.getString(key);
    }
1

}


