package com.zork.dragoncontrol.component;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.zork.common.dto.MetricDTO;
import com.zork.dragoncontrol.constant.DragonControlConstants;
import com.zork.dragoncontrol.model.HisDate;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.*;
import org.springframework.jdbc.core.BeanPropertyRowMapper;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import static com.zork.common.utils.DateUtil.yearMonthDayDigital;
import static com.zork.common.utils.FileUtil.readFile;
import static com.zork.common.utils.ZorkUtil.getLocalIp;
import static com.zork.common.utils.ZorkUtil.getPinYinHeadChar;
import static com.zork.dragoncontrol.constant.DragonControlConstants.*;
import static com.zork.dragoncontrol.constant.DragonControlConstants.ID;


/**
 * @Author: Prock.Liy
 * @Date： 2021/6/1
 * @Description： 查询龙控系统指标组件
 */
@Slf4j
@Component
public class DataBaseQueryComponent {

    @Value("${send.metric-url}")
    private String sendMetricUrl;

    @Value("${query-sql}")
    private String querySql;

    @Value("${send.query-file}")
    private String queryFile;

    @Value("${send.metric-JMKT}")
    private String sendmetricMKT;

    @Resource
    private RestTemplate restTemplate;

    @Resource
    @Qualifier("jdbcTemplateCenter")
    private JdbcTemplate jdbcTemplateCenter;

    /**
     * thread pools   如果线程池长度超过处理需要，可灵活回收空闲线程，若无可回收，则新建线程
     */
    protected ExecutorService EXECUTOR_POOL = Executors.newCachedThreadPool();

    private static int num = 0;

    /**
     * 根据.json配置文件查询库中的指定指标数据
     */
    @Scheduled(cron = "${schedules}")
    public void taskExecute() {
        // 解析配置文件
        JSONArray queryConfigArray = JSON.parseArray(readFile(queryFile));
        // 拼接Sql查询数据后处理
        //analyzeMetricConfig(queryConfigArray);
        analyzeMetric(queryConfigArray);

    }

    public void execute(String sql, List<String> IDList, String metricsetname) {
        try {
            EXECUTOR_POOL.execute(() -> {
                // 查询最近一分钟内数据,由于表数据量比较大，需要设置连接超时时间
                List<HisDate> hisDateList = jdbcTemplateCenter.query(sql, new BeanPropertyRowMapper<HisDate>(HisDate.class));
                // MgrObjId去重
                //System.out.println(hisDateList.toString());
                List<String> mgrObjIdList = hisDateList.stream()
                        .filter(distinctByKey(HisDate::getMgrObjId))
                        .map(HisDate::getMgrObjId)
                        .collect(Collectors.toList());
                //System.out.println(mgrObjIdList.toString());
                // 根据JSON配置文件解析数据，去重后获取最新时间的数据

                MetricDTO metricDTO = new MetricDTO();
                metricDTO.setTimestamp(new Date().getTime());
                metricDTO.setMetricsetname(metricsetname);
                Map<String, String> measures = new HashMap<>();
                Map<String, String> dimensions = new HashMap<>(3);
                dimensions.put("appsystem","LKsystem");
                dimensions.put("appprogramname",metricsetname);
                dimensions.put("ip",getLocalIp());

                for (String objId : mgrObjIdList) {
                    IDList.forEach(ID -> {
                        Optional<HisDate> latestTimeHisDate = hisDateList.stream()
                                .filter(hisDate -> hisDate.getId().equals(ID) && hisDate.getMgrObjId().equals(objId))
                                 .max(Comparator.comparing(HisDate::getDtime));
                        //System.out.println(latestTimeHisDate);
                        // 不为空则拼装measures
                        if (latestTimeHisDate.isPresent()){
                            measures.put(ID, latestTimeHisDate.get().getValue());
                            dimensions.put(DEV_NAME,latestTimeHisDate.get().getDevName().toUpperCase());
                            dimensions.put(MGROBJID,latestTimeHisDate.get().getMgrObjId().toUpperCase());
                        }
                        latestTimeHisDate.ifPresent(hisDate -> measures.put(ID, hisDate.getValue()));
                    });
                    metricDTO.setDimensions(dimensions);
                    metricDTO.setMeasures(measures);

                    // 发送指标数据至Kafka接口
                    //System.out.println(metricDTO);
                    sendMetric2Kafka(metricDTO);
                }
            });
        } catch (Exception e) {
            log.error("\n Query MetricDataExecute: -> {}", e.getMessage());
        }
    }

    /**
     * 根据配置文件拼接出sql调用execute查询数据后处理
     *
     * @return
     */
    public void analyzeMetricConfig(JSONArray queryConfigArray) {
        try {
            queryConfigArray.forEach(object -> {
                List<String> IDList = new ArrayList<>();
                JSONObject metric = (JSONObject) object;
                JSONArray measures = metric.getJSONArray(MEASURES);
                String metricsetname = metric.getString(METRIC_SET_NAME);
                StringBuilder idSql = new StringBuilder();

                if (measures.size() > 0) {
                    idSql.append("AND his.Id IN(");
                }
                measures.forEach(measure -> {
                    if (num != 0) {
                        idSql.append(",");
                    }
                    JSONObject jsonObject = (JSONObject) measure;
                    idSql.append("'").append(jsonObject.get(ID)).append("'");
                    IDList.add(jsonObject.get(ID).toString());
                    num++;
                });
                idSql.append(")");

                // 查询数据后处理
                execute(String.format(querySql, yearMonthDayDigital(), metric.getString(AGENT_BM), idSql.toString()), IDList, metricsetname);
            });
        } catch (Exception e) {
            log.error("\n analyzeMetricConfig Execute: -> {}", e.getMessage());
        } finally {
            num = 0;
        }
    }
    public void analyzeMetric(JSONArray queryConfigArray) {
        try {

            queryConfigArray.forEach(object -> {
            List<String> IDList = new ArrayList<>();
            JSONObject metric = (JSONObject) object;
            JSONArray measures = metric.getJSONArray(MEASURES);
            String metricsetname = metric.getString(METRIC_SET_NAME);
            //StringBuilder idSql = new StringBuilder();
            String idSql = metric.getString("sql");
                measures.forEach(measure -> {
                    JSONObject jsonObject = (JSONObject) measure;
                    IDList.add(jsonObject.get(ID).toString());
                });
            // 查询数据后处理
             execute(String.format(idSql, yearMonthDayDigital()), IDList, metricsetname);
            });
        } catch (Exception e) {
            log.error("\n analyzeMetricConfig Execute: -> {}", e.getMessage());
        } finally {
            num = 0;
        }
    }
    /**
     * 根据特定值去重
     *
     * @param keyExtractor
     * @param <T>
     * @return
     */
    public static <T> Predicate<T> distinctByKey(Function<? super T, Object> keyExtractor) {
        Map<Object, Boolean> map = new ConcurrentHashMap<>();
        return t -> map.putIfAbsent(keyExtractor.apply(t), Boolean.TRUE) == null;
    }

//    public static void main(String[] args) {
//        List<HisDate> list = new ArrayList<>();
//        list.add(HisDate.builder().Id("temp1").Value("1").MgrObjId("1").DevName("q1").build());
//        list.add(HisDate.builder().Id("temp2").Value("2").MgrObjId("1").DevName("q1").build());
//
//        list.add(HisDate.builder().Id("temp1").Value("3").MgrObjId("2").DevName("q2").build());
//        list.add(HisDate.builder().Id("temp2").Value("4.0").MgrObjId("2").DevName("q2").build());
//
//        list.add(HisDate.builder().Id("temp1").Value("0").MgrObjId("1").DevName("q1").build());
//        list.add(HisDate.builder().Id("temp2").Value("0").MgrObjId("1").DevName("q1").build());
//
//        // 根据ID进行去重
//        List<String> resultAtootaList = list.stream()
//                .filter(distinctByKey(HisDate::getMgrObjId))
//                .map(HisDate::getMgrObjId)
//                .collect(Collectors.toList());
//        System.out.println(resultAtootaList.size());
//    }

    /**
     * 发送指标数据至Kafka
     *
     * @param metricDTO
     * @return JSONObject
     */
    private void sendMetric2Kafka(MetricDTO metricDTO) {
        try {
            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.APPLICATION_JSON);
            HttpEntity entity = new HttpEntity<>(JSONObject.toJSON(metricDTO), headers);
            ResponseEntity<String> result = restTemplate.exchange(sendMetricUrl, HttpMethod.POST, entity, String.class);

            // 调用失败缺少measure值时，进行补齐重新调归补用递
            if (!Objects.requireNonNull(result.getBody()).contains("OK")) {
                log.info("sendMetric2Kafka SUCCESS,调用参数: ->{}  ,接口调用Result: ->{}", JSONObject.toJSON(metricDTO), result.getBody());
            }
        } catch (Exception e) {
            log.error("sendMetric2Kafka Exception: -> {} ,调用参数: ->{}", e.getMessage(), JSONObject.toJSON(metricDTO));
        }
    }

}
