package com.zork.junhiro.trading.count.component;


import com.zork.common.dto.ExecuteResultDTO;
import com.zork.junhiro.trading.count.config.InfluxDBConfig;
import com.zork.junhiro.trading.count.model.app.YesterDayDataDO;
import com.zork.junhiro.trading.count.model.influx.OnLineData;
import com.zork.junhiro.trading.count.model.msdb.SysConfigDO;
import com.zork.common.service.InfluxDBService;
import com.zork.common.utils.DateUtil;
import com.zork.common.utils.SqlFileExecuteUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.influxdb.dto.QueryResult;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.dao.support.DataAccessUtils;
import org.springframework.jdbc.core.BeanPropertyRowMapper;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.scheduling.annotation.Schedules;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.*;

import static com.zork.common.constant.InfluxDBConstant.AUTOGEN;
import static com.zork.common.utils.FileUtil.judgeTradingDay;
import static com.zork.common.utils.FileUtil.readFile;
import static com.zork.junhiro.trading.count.constant.PeakConstant.*;
import static com.zork.common.service.InfluxDBService.getResult;
import static com.zork.common.utils.DateUtil.dateToStamp;
import static com.zork.common.utils.DateUtil.parseTime;

/**
 * @Author: Prock.Liy
 * @Date： 2021/6/1
 * @Description： 当天交易日数据统计
 */
@Slf4j
@Component
public class StatisticsComponent {

    @Value("${spring.datasource.msdb.url}")
    private String url;

    @Value("${spring.datasource.msdb.username}")
    private String username;

    @Value("${spring.datasource.msdb.password}")
    private String password;

    @Value("${am.start-time}")
    private String amStartTime;

    @Value("${am.end-time}")
    private String amEndTime;

    @Value("${pm.end-time}")
    private String pmEndTime;

    @Value("${am.detect.start-time}")
    private String amDetectStartTime;

    @Value("${am.detect.end-time}")
    private String amDetectEndTime;

    @Value("${pm.detect.start-time}")
    private String pmDetectStartTime;

    @Value("${pm.detect.end-time}")
    private String pmDetectEndTime;

    @Value("${sql.jhapp-file}")
    private String jhAppFile;

    @Value("${trading-day.filePath}")
    private String tradingDayFilePath;

    @Resource
    @Qualifier("jdbcTemplateMsdb")
    private JdbcTemplate jdbcTemplateMsdb;

    @Resource
    @Qualifier("jdbcTemplateAppBusiness")
    private JdbcTemplate jdbcTemplateAppBusiness;

    @Resource
    private InfluxDBConfig influxDBConfig;

    /**
     * 保留double两位小数格式化
     */
    private final DecimalFormat decimalFormat = new DecimalFormat("0.0");

    private final SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyyMMdd");

    private final SimpleDateFormat tadingFormat = new SimpleDateFormat("yyyy.MM.dd");


    /**
     * 连接数据库  根据cron表达式am pm执行
     */
    @Schedules({
            @Scheduled(cron = "${schedules-Junhiro.am}"),
            @Scheduled(cron = "${schedules-Junhiro.pm}"),
            @Scheduled(cron = "${schedules-Junhiro.detect-am}"),
            @Scheduled(cron = "${schedules-Junhiro.detect-pm}")
    })
    public void JunHiroExecute() {
        try {
            // 判断是否是交易日
            String today = tadingFormat.format(new Date());
            // 不为交易日直接退出
            if (!judgeTradingDay(tradingDayFilePath, today)) {
                return;
            }
            SysConfigDO sysConfigDO = jdbcTemplateMsdb.queryForObject("SELECT sysdate,lastsysdate FROM run..sysconfig WHERE serverid = 0", new BeanPropertyRowMapper<>(SysConfigDO.class));

            // 从数据库获取截至到3点半上个交易日数据
            List<YesterDayDataDO> yesterDayDataList = jdbcTemplateAppBusiness.query("SELECT TOP 1 * FROM  YesterDayData  where yesterday = ? ORDER BY id DESC", new BeanPropertyRowMapper<YesterDayDataDO>(YesterDayDataDO.class), sysConfigDO.getLastSysdate());
            YesterDayDataDO yesterDayDataDO = DataAccessUtils.uniqueResult(yesterDayDataList);

//            SysConfigDO sysConfigDO = new SysConfigDO();
//            sysConfigDO.setLastSysdate(20220902);
//            YesterDayDataDO yesterDayDataDO = new YesterDayDataDO();
//            yesterDayDataDO.setYesterday(1);
//            yesterDayDataDO.setQuotesPeak(14.1);
//            yesterDayDataDO.setContent("test");

            String startTime = StringUtils.EMPTY;
            String endTime = StringUtils.EMPTY;
            String period = StringUtils.EMPTY;
            if (new Date().getHours() >= 15) {
                startTime = amStartTime;
                endTime = pmEndTime;
                period = PM;
                log.info("开始查询今日下午数据");
            } else if (new Date().getHours() == 14){
                startTime = pmDetectStartTime;
                endTime = pmDetectEndTime;
                period = PM;
                log.info("开始探测今日下午数据");
            } else if (new Date().getHours() >= 9) {
                startTime = amStartTime;
                endTime = amEndTime;
                period = AM;
                log.info("开始查询今日上午数据");
            } else if (new Date().getHours() == 8) {
                startTime = amDetectStartTime;
                endTime = amDetectEndTime;
                period = AM;
                log.info("开始探测今日上午数据");
            } else {
                startTime = amStartTime;
                endTime = pmEndTime;
                period = PM;
                log.info("查询今日9-15点数据");
            }

            // 获取今日上午指定时间段内数据
            YesterDayDataDO quotes = getPeakData(QUOTES_PEAK, sysConfigDO.getLastSysdate(), yesterDayDataDO, startTime, endTime, period);
            if (StringUtils.isEmpty(quotes.getContent())) {
                log.error("QuotesPeak Exception:{}", "获取今日行情用户在线数与峰值无数据");
                return;
            }
            YesterDayDataDO transaction = getPeakData(TRANSACTION_PEAK, sysConfigDO.getLastSysdate(), yesterDayDataDO, startTime, endTime, period);
            if (StringUtils.isEmpty(transaction.getContent())) {
                log.error("TransactionPeak Exception:{}", "获取今日交易用户在线数与峰值无数据");
                return;
            }
//            log.info(quotes.getContent() + transaction.getContent());
            System.out.println(quotes.getContent() + transaction.getContent());
            String todaydata = quotes.getContent() + transaction.getContent();
            String todaydate = new SimpleDateFormat("yyyy-MM-dd").format(new Date());

//            jdbcTemplateAppBusiness.update(
//                    "INSERT INTO jhlog (date,jhstring) VALUES (?,?);",
//                    todaydate,
//                    todaydata
//
//            );


        } catch (Exception e) {
            e.printStackTrace();
            log.error("Execute Exception:{}", e.getMessage());
        }
    }


    /**
     * 连接数据库  闭市后15点30后执行
     */
    @Scheduled(cron = "${schedules}")
    public void execute() {
        try {
            // 判断是否是交易日
            String today = tadingFormat.format(new Date());
            // 不为交易日直接退出
            if (!judgeTradingDay(tradingDayFilePath, today)) {
                return;
            }
            StringBuffer content = new StringBuffer();

            SysConfigDO sysConfigDO = jdbcTemplateMsdb.queryForObject("SELECT sysdate,lastsysdate FROM run..sysconfig WHERE serverid = 0", new BeanPropertyRowMapper<>(SysConfigDO.class));
            log.info("今日时间为:%s    上一交易日时间为:%s", sysConfigDO.getSysdate(), sysConfigDO.getLastSysdate());

            // 判断当天是否为交易日
            List<Integer> bizFlagList = jdbcTemplateAppBusiness.queryForList("SELECT TOP 1 bizflag FROM TradingDay WHERE date = ?", Integer.class, sysConfigDO.getSysdate());
            Integer bizFlag = DataAccessUtils.uniqueResult(bizFlagList);
            if (bizFlag == 1) {
                log.info("今日不是交易日,查询结束");
                return;
            }
            // 当天为周一便查询上周五交易日数据
            if (DateUtil.getWeek() == 1) {
                sysConfigDO.setLastSysdate(sysConfigDO.getLastSysdate() - 3);
                System.out.println("周一时间更换后未:" + sysConfigDO.getLastSysdate());
            }

            // 从数据库获取截至到3点半上个交易日数据
            List<YesterDayDataDO> yesterDayDataList = jdbcTemplateAppBusiness.query("SELECT TOP 1 * FROM  YesterDayData  where yesterday = ? ORDER BY id DESC", new BeanPropertyRowMapper<YesterDayDataDO>(YesterDayDataDO.class), sysConfigDO.getLastSysdate());
            YesterDayDataDO yesterDayDataDO = DataAccessUtils.uniqueResult(yesterDayDataList);

            YesterDayDataDO quotes = getPeakData(QUOTES_PEAK, sysConfigDO.getLastSysdate(), yesterDayDataDO, amStartTime, "T09:59:00Z", PM);
            if (StringUtils.isEmpty(quotes.getContent())) {
                log.error("QuotesPeak Exception:{}", "获取今日用户在线数与峰值无数据");
                return;
            }
            content.append(quotes.getContent());
            YesterDayDataDO transaction = getPeakData(TRANSACTION_PEAK, sysConfigDO.getLastSysdate(), yesterDayDataDO, amStartTime, "T09:59:00Z", PM);
            if (StringUtils.isEmpty(transaction.getContent())) {
                log.error("TransactionPeak Exception:{}", "获取今日用户在线数与峰值无数据");
                return;
            }
            content.append(transaction.getContent());

            ExecuteResultDTO numData = SqlFileExecuteUtil.executeSql("11", url, username, password, jhAppFile);
            if (numData.getStatus() == 1) {
                return;
            }
            YesterDayDataDO centralized = getCentralizedData(numData.getLog(), yesterDayDataDO);
            if (StringUtils.isEmpty(centralized.getContent())) {
                log.error("getCentralizedData Exception:{}", "获取今日用户在线数与峰值无数据");
            }
            System.out.println(centralized.getContent() + quotes.getContent() + transaction.getContent());

            // 数据库存储今日数据
            jdbcTemplateAppBusiness.update(
                    "INSERT INTO YesterDayData (yesterday,base_stock_sum,traffic_peak,quotes_peak,transaction_peak) VALUES (?,?,?,?,?);",
                    sysConfigDO.getSysdate(),
                    centralized.getBaseStockSum(),
                    centralized.getTrafficPeak(),
                    quotes.getQuotesPeak(),
                    transaction.getTransactionPeak()
            );
        } catch (Exception e) {
            e.printStackTrace();
            log.error("Execute Exception:{}", e.getMessage());
        }
    }

    /**
     * 打印交易人数,四舍五入w
     */
    @Scheduled(cron = "${peak-cron.ten}")
    public void peakTen() {
        try {
            // 行情在线查询
            OnLineData quotes = getOnLineData(QUOTES_COLUMN, QUOTES_TABLE_NAME,
                    dateToStamp(new SimpleDateFormat("yyyy-MM-dd").format(new Date()) + "T09:00:00Z"),
                    dateToStamp(new SimpleDateFormat("yyyy-MM-dd").format(new Date()) + pmEndTime), PM, true
            );
            OnLineData trade = getOnLineData(TRANSACTION_COLUMN, TRANSACTION_TABLE_NAME,
                    dateToStamp(new SimpleDateFormat("yyyy-MM-dd").format(new Date()) + "T09:00:00Z"),
                    dateToStamp(new SimpleDateFormat("yyyy-MM-dd").format(new Date()) + pmEndTime), PM, true
            );
            StringBuilder data = new StringBuilder();
            data.append("君弘行情在线峰值").append(quotes.getSum() + "万人，");
            data.append("交易在线峰值").append(trade.getSum() + "万人。");
            //System.out.println(data.toString());
        } catch (Exception e) {
            e.printStackTrace();
            log.error("peakTen Exception:{}", e.getMessage());
        }
    }

    /**
     * 打印行情交易人数
     */
    @Scheduled(cron = "${peak-cron.people-num}")
    public void peakPeopleNum() {
        try {
            // 行情在线查询
            OnLineData quotes = getOnLineData(QUOTES_COLUMN, QUOTES_TABLE_NAME,
                    dateToStamp(new SimpleDateFormat("yyyy-MM-dd").format(new Date()) + "T09:00:00Z"),
                    dateToStamp(new SimpleDateFormat("yyyy-MM-dd").format(new Date()) + pmEndTime), PM, false
            );
            OnLineData trade = getOnLineData(TRANSACTION_COLUMN, TRANSACTION_TABLE_NAME,
                    dateToStamp(new SimpleDateFormat("yyyy-MM-dd").format(new Date()) + "T09:00:00Z"),
                    dateToStamp(new SimpleDateFormat("yyyy-MM-dd").format(new Date()) + pmEndTime), PM, false
            );
            StringBuilder data = new StringBuilder();
            data.append("君弘行情在线峰值").append((int) quotes.getSum() + "人，");
            data.append("交易在线峰值").append((int) trade.getSum() + "人。");
            //System.out.println(data.toString());
        } catch (Exception e) {
            e.printStackTrace();
            log.error("peakPeopleNum Exception:{}", e.getMessage());
        }
    }


    /**
     * 获取今日在线数与峰值
     *
     * @param lastSysdate
     * @return
     */
    public YesterDayDataDO getPeakData(String QueryType, Integer lastSysdate, YesterDayDataDO yesterDayDataDO,
                                       String startTime, String endTime, String period) {
        YesterDayDataDO dayDataDO = new YesterDayDataDO();
        StringBuilder data = new StringBuilder();
        try {
            String column = StringUtils.EMPTY;
            String tableName = StringUtils.EMPTY;
            switch (QueryType) {
                case QUOTES_PEAK:
                    column = QUOTES_COLUMN;
                    tableName = QUOTES_TABLE_NAME;
                    break;
                case TRANSACTION_PEAK:
                    column = TRANSACTION_COLUMN;
                    tableName = TRANSACTION_TABLE_NAME;
                    break;
                default:
                    break;
            }
            // 今日数据
            OnLineData nowadays = getOnLineData(column, tableName,
                    dateToStamp(new SimpleDateFormat("yyyy-MM-dd").format(new Date()) + startTime),
                    dateToStamp(new SimpleDateFormat("yyyy-MM-dd").format(new Date()) + endTime), period, true
            );

            double yesPeak = 0.0;
            // 数据库中无数据时，从InfoDB查询上个交易日数据
            if (Objects.isNull(yesterDayDataDO)) {
                OnLineData yesterday = getOnLineData(column, tableName,
                        dateToStamp(new SimpleDateFormat("yyyy-MM-dd").format(simpleDateFormat.parse(lastSysdate.toString())) + startTime),
                        dateToStamp(new SimpleDateFormat("yyyy-MM-dd").format(simpleDateFormat.parse(lastSysdate.toString())) + endTime), period, true
                );
                if (Objects.isNull(yesterday)) {
                    log.error("InfluxDB获取上一交易日数据为空 时间为:{}", lastSysdate);
                    return dayDataDO;
                }
                yesPeak = getDouble1f(yesterday.getSum());
            } else {
                yesPeak = getDouble1f(yesterDayDataDO.getQuotesPeak());
            }


            switch (QueryType) {
                case QUOTES_PEAK:
                    dayDataDO.setQuotesPeak(yesPeak);
                    if (AM.equals(period)) {
                        //data.append("君弘行情在线").append(nowadays.getSum())
                        data.append("今日" + new SimpleDateFormat("yyyy-MM-dd").format(new Date()) + " 09:30")
                                .append("，君弘行情在线").append(nowadays.getSum())
                                .append("万人（昨日开盘" + yesPeak + "万人，")
                                .append(countPercentage(nowadays.getSum(), yesPeak))
                                .append("，在线数占总容量" + decimalFormat.format((nowadays.getSum() / 125.3) * 100) + "%），");
                    } else if (PM.equals(period)) {
                        data.append("，君弘行情在线峰值").append(nowadays.getSum())
                                .append("万人（昨日峰值" + yesPeak + "万人，")
                                .append(countPercentage(nowadays.getSum(), yesPeak))
                                .append("），");
                    }
                    break;
                case TRANSACTION_PEAK:
                    dayDataDO.setTransactionPeak(yesPeak);
                    if (AM.equals(period)) {
                        data.append("交易在线").append(nowadays.getSum())
                                .append("万人（昨日开盘" + yesPeak + "万人，")
                                .append(countPercentage(nowadays.getSum(), yesPeak))
                                .append("，在线数占总容量" + decimalFormat.format((nowadays.getSum() / 253) * 100) + "%）。");
                    } else if (PM.equals(period)) {
                        data.append("交易在线峰值").append(nowadays.getSum())
                                .append("万人（昨日峰值" + yesPeak + "万人，")
                                .append(countPercentage(nowadays.getSum(), yesPeak))
                                .append("）。");
                    }
                    break;
                default:
                    break;
            }

        } catch (Exception e) {
            e.printStackTrace();
            log.error("getPeak Exception:{}", e.getMessage());
        }
        return dayDataDO.setContent(data.toString());
    }

    /**
     * 获取今日集中交易系统统计数据
     *
     * @param databaseResult  数据库查询结果
     * @param yesterDayDataDO 上交易日结果集
     * @return
     */
    public YesterDayDataDO getCentralizedData(String databaseResult, YesterDayDataDO yesterDayDataDO) {
        YesterDayDataDO dayDataDO = new YesterDayDataDO();
        StringBuffer centralize = new StringBuffer("今日公司集中交易系统委托笔数");
        try {
            List<String> centralizeList = Arrays.asList(
                    StringUtils.substringAfterLast(StringUtils.substringBeforeLast(databaseResult.replaceAll("\r", ""), "\n"), "\n").split("\t")
            );
            if (centralizeList.size() != 9) {
                log.info("执行sql文件结果: ---> {}", databaseResult);
                return dayDataDO;
            }
            double ntoDayOrDerNum = Double.parseDouble(centralizeList.get(0));
            double ntoDayMatchNum = Double.parseDouble(centralizeList.get(1));
            double ntoDayMatchMt = Double.parseDouble(centralizeList.get(2));
            dayDataDO.setBaseStockSum(ntoDayMatchMt);
            // 昨日数据为空则从SqlServer获取
            double nLastDayMatchAmt = !Objects.isNull(yesterDayDataDO) ? yesterDayDataDO.getBaseStockSum() : Double.parseDouble(centralizeList.get(3));

            String time = centralizeList.get(5);
            double peak = Double.parseDouble(centralizeList.get(6));
            dayDataDO.setTrafficPeak(peak);
            double yesPeak = !Objects.isNull(yesterDayDataDO) ? yesterDayDataDO.getTrafficPeak() : Double.parseDouble(centralizeList.get(8));

            // 字符串数据拼接
            centralize.append(getInt(ntoDayOrDerNum)).append("万笔，成交笔数").append(getInt(ntoDayMatchNum)).append("万笔，");
            centralize.append("股基成交金额").append(getInt(ntoDayMatchMt)).append("亿（上交易日股基成交金额").append(getInt(nLastDayMatchAmt)).append("亿，").append(countPercentage(ntoDayMatchMt, nLastDayMatchAmt)).append("）。");
            centralize.append("集中交易系统综合业务量在").append(time).append("达到峰值").append(decimalFormat.format(peak)).append("万笔/秒，较上一交易日").append(countPercentage(peak, yesPeak)).append(" ");
            centralize.append("（上一交易日").append(decimalFormat.format(yesPeak)).append("万笔/秒）。");
        } catch (Exception e) {
            e.printStackTrace();
            log.error("getCentralizedData Exception:  " + e.getMessage());
        }
        return dayDataDO.setContent(centralize.toString());
    }

    /**
     * 计算百分比
     *
     * @param toDayData
     * @param yesData
     * @return
     */
    private String countPercentage(double toDayData, double yesData) {
        String result = StringUtils.EMPTY;
        if (toDayData > yesData) {
            result = "同比增长" + decimalFormat.format((toDayData - yesData) / yesData * 100) + "%";
        } else if (toDayData < yesData) {
            result = "同比下降" + decimalFormat.format((yesData - toDayData) / yesData * 100) + "%";
        } else {
            result = "同比持平";
        }
        return result;
    }

    /**
     * 从influxDB中查询9:00-9:59的峰值数据
     *
     * @param column    统计列名
     * @param column    数据库表名
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @return JhAppHqOnLine
     */
    public OnLineData getOnLineData(String column, String tableName, long startTime, long endTime, String period, boolean isDouble) {
        try {

            String querySql = "select sum(\"" + column + "\") from \"default\".\"" + tableName + "\" " +
                    "WHERE time>=" + startTime + "ms AND time<=" + endTime + "ms GROUP BY time(30s) fill(none) tz('Asia/Shanghai')";
            log.info("InfluxDB Execute Command: {}", querySql);
            QueryResult result = influxDBConfig.getInfluxDBService().query(querySql);
            QueryResult.Series series = getResult(result);
            List<OnLineData> onLineList = new ArrayList<>();

            // 取值
            if (series.getValues().isEmpty()){
                return new OnLineData();
            }

            for (List<Object> value : series.getValues()){
                String time = parseTime(value.get(0).toString());
                double sum;
                if (isDouble) {
                    sum = (double) Math.round(Double.parseDouble(value.get(1).toString()) / 10000 * 100) / 100;
                } else {
                    sum = (double) Math.round(Double.parseDouble(value.get(1).toString()));
                }

                // 如果是上午的话，查询至31分的数据；解决延迟数据不对问题
                if (period.equals(AM)){
                    assert time != null;
                    if (!time.contains("09:30:00")){
                        continue;
                    }
                }
                onLineList.add(
                        new OnLineData()
                                .setTime(time)
                                .setSum(sum)
                );
            }


            Optional<OnLineData> optional = onLineList.stream().max(Comparator.comparingDouble(OnLineData::getSum));
            return optional.orElse(null);
        } catch (Exception e) {
            log.error("InfluxDBDataConversion Exception:{}", e.getMessage());
            return null;
        }
    }

    /**
     * 四舍五入把double转化int整型，0.5进一，小于0.5不进一
     *
     * @param number number
     * @return int
     */
    public static int getInt(double number) {
        BigDecimal bd = new BigDecimal(number).setScale(0, BigDecimal.ROUND_HALF_UP);
        return Integer.parseInt(bd.toString());
    }

    public static double getDouble1f(double data) {
        return new BigDecimal(data).setScale(1, BigDecimal.ROUND_HALF_UP).doubleValue();
    }


}
