package com.zork.disorder.component;

import lombok.extern.slf4j.Slf4j;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @Author: Optimized
 * @Date: 2024
 * @Description: 智能去重组件，支持多库多表场景，内存可控
 */
@Slf4j
@Component
public class SmartDeduplicator {
    
    // 存储去重键和时间戳
    private final Map<String, Long> keyTimestamps = new ConcurrentHashMap<>();
    
    // 最大缓存数量，防止内存爆炸
    private static final int MAX_KEYS = 100000;
    // 数据有效期（5分钟）
    private static final long VALIDITY_PERIOD = 300000;
    
    /**
     * 检查数据是否重复
     * 
     * @param url InfluxDB地址
     * @param database 数据库名
     * @param tableName 表名
     * @param id 数据ID
     * @return true-重复，false-新数据
     */
    public boolean isDuplicate(String url, String database, String tableName, Long id) {
        if (id == null) {
            return false; // null ID不去重
        }
        
        String key = buildDeduplicationKey(url, database, tableName, id);
        long now = System.currentTimeMillis();
        
        // 检查是否存在且在有效期内
        Long timestamp = keyTimestamps.get(key);
        if (timestamp != null && (now - timestamp) < VALIDITY_PERIOD) {
            return true; // 重复数据
        }
        
        // 缓存大小控制
        if (keyTimestamps.size() >= MAX_KEYS) {
            cleanupExpiredKeys(now);
            
            // 如果清理后还是太多，强制清理一半
            if (keyTimestamps.size() >= MAX_KEYS) {
                forceCleanup();
            }
        }
        
        // 添加新键
        keyTimestamps.put(key, now);
        return false; // 新数据
    }
    
    /**
     * 构建去重键：host|database|table|id
     */
    private String buildDeduplicationKey(String url, String database, String tableName, Long id) {
        String host = extractHost(url);
        return String.format("%s|%s|%s|%d", host, database, tableName, id);
    }
    
    /**
     * 从URL中提取主机地址
     */
    private String extractHost(String url) {
        try {
            // http://influx1:8086 -> influx1:8086
            return url.substring(url.indexOf("://") + 3);
        } catch (Exception e) {
            return url; // 解析失败返回原值
        }
    }
    
    /**
     * 清理过期的键
     */
    private void cleanupExpiredKeys(long now) {
        int beforeSize = keyTimestamps.size();
        keyTimestamps.entrySet().removeIf(entry -> 
            (now - entry.getValue()) > VALIDITY_PERIOD);
        int afterSize = keyTimestamps.size();
        
        if (beforeSize - afterSize > 0) {
            log.debug("清理过期去重键 {} 个，剩余 {} 个", beforeSize - afterSize, afterSize);
        }
    }
    
    /**
     * 强制清理缓存（保留最新的一半）
     */
    private void forceCleanup() {
        int targetSize = MAX_KEYS / 2;
        int beforeSize = keyTimestamps.size();
        
        // 按时间戳排序，清理旧的数据
        keyTimestamps.entrySet().stream()
            .sorted(Map.Entry.comparingByValue())
            .limit(beforeSize - targetSize)
            .forEach(entry -> keyTimestamps.remove(entry.getKey()));
            
        log.warn("内存压力过大，强制清理去重缓存 {} 个，剩余 {} 个", 
            beforeSize - keyTimestamps.size(), keyTimestamps.size());
    }
    
    /**
     * 定时清理任务
     */
    @Scheduled(fixedDelay = 300000) // 5分钟执行一次
    public void scheduleCleanup() {
        long now = System.currentTimeMillis();
        int beforeSize = keyTimestamps.size();
        
        cleanupExpiredKeys(now);
        
        int afterSize = keyTimestamps.size();
        if (beforeSize - afterSize > 0) {
            log.info("定时清理去重缓存 {} 个，当前缓存数量: {}", beforeSize - afterSize, afterSize);
        }
    }
    
    /**
     * 获取缓存统计信息
     */
    public String getCacheStats() {
        return String.format("去重缓存数量: %d / %d", keyTimestamps.size(), MAX_KEYS);
    }
    
    /**
     * 手动清理所有缓存
     */
    public void clearAll() {
        int size = keyTimestamps.size();
        keyTimestamps.clear();
        log.info("手动清理所有去重缓存 {} 个", size);
    }
}