package com.zork.disorder.component;

import com.zork.common.dto.NormalFieldsDTO;
import com.zork.disorder.kafka.KafkaSendDTO;
import com.zork.disorder.kafka.Producer;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;

/**
 * @Author: Optimized
 * @Date: 2024
 * @Description: 批量Kafka发送器，收集数据批量发送提高性能
 */
@Slf4j
@Component
public class BatchKafkaSender {
    
    @Value("${spring.kafka.bootstrap-servers}")
    private String bootstrapServers;
    
    @Value("${spring.kafka.producer.topic}")
    private String topic;
    
    
    @Value("${batch.size:1000}")
    private int batchSize;
    
    @Value("${batch.timeout:2000}")
    private long batchTimeoutMs;
    
    // 高性能无锁队列缓冲区
    private final ConcurrentLinkedQueue<NormalFieldsDTO> buffer = new ConcurrentLinkedQueue<>();
    
    // 发送状态控制，避免并发发送
    private final AtomicBoolean isSending = new AtomicBoolean(false);
    
    // Kafka Producer 实例
    private Producer producer;
    
    // 线程安全统计信息
    private final AtomicLong totalSent = new AtomicLong(0);
    private final AtomicLong totalBatches = new AtomicLong(0);
    private volatile long lastSendTime = System.currentTimeMillis();
    
    /**
     * 初始化 Kafka Producer
     */
    @PostConstruct
    public void init() {
        KafkaSendDTO kafkaSendDTO = new KafkaSendDTO();
        kafkaSendDTO.setBootstrapServers(bootstrapServers);
        kafkaSendDTO.setTopic(topic);
        this.producer = new Producer(kafkaSendDTO);
        log.info("BatchKafkaSender 初始化完成，批次大小: {}, 超时: {}ms", batchSize, batchTimeoutMs);
    }
    
    /**
     * 添加数据到批量缓冲区
     * 
     * @param data 要发送的数据列表
     */
    public void addData(List<NormalFieldsDTO> data) {
        if (data == null || data.isEmpty()) {
            return;
        }
        
        // 无锁添加数据
        data.forEach(buffer::offer);
        log.debug("添加 {} 条数据到缓冲区，当前大小: {}", data.size(), buffer.size());
        
        // 非阻塞检查是否需要发送
        tryFlushIfNeeded();
    }
    
    /**
     * 添加单条数据到批量缓冲区
     */
    public void addData(NormalFieldsDTO data) {
        if (data == null) {
            return;
        }
        
        buffer.offer(data);
        tryFlushIfNeeded();
    }
    
    /**
     * 非阻塞检查是否需要刷新
     */
    private void tryFlushIfNeeded() {
        int size = buffer.size();
        long timeSinceLastSend = System.currentTimeMillis() - lastSendTime;
        
        if (size >= batchSize || (size > 0 && timeSinceLastSend > batchTimeoutMs)) {
            flushBuffer(size >= batchSize ? "达到批次大小" : "超时刷新");
        }
    }
    
    /**
     * 高性能无锁刷新缓冲区
     */
    public void flushBuffer(String reason) {
        if (buffer.isEmpty() || !isSending.compareAndSet(false, true)) {
            return; // 空或正在发送，直接返回
        }
        
        try {
            List<NormalFieldsDTO> toSend = new ArrayList<>();
            
            // 快速无锁抽取数据
            for (int i = 0; i < batchSize && !buffer.isEmpty(); i++) {
                NormalFieldsDTO data = buffer.poll();
                if (data != null) {
                    toSend.add(data);
                }
            }
            
            if (!toSend.isEmpty()) {
                sendBatch(toSend, reason);
            }
            
        } finally {
            isSending.set(false);
        }
    }
    
    /**
     * 批量发送数据到Kafka
     */
    private void sendBatch(List<NormalFieldsDTO> batch, String reason) {
        if (batch.isEmpty()) {
            return;
        }
        
        long startTime = System.currentTimeMillis();
        
        try {
            // 使用复用的Producer异步发送
            producer.sendMessage(batch).whenComplete((result, throwable) -> {
                if (throwable == null) {
                    // 线程安全更新统计信息
                    long newTotal = totalSent.addAndGet(batch.size());
                    totalBatches.incrementAndGet();
                    lastSendTime = System.currentTimeMillis();
                    
                    long duration = System.currentTimeMillis() - startTime;
                    log.info("异步发送成功 - 原因: {}, 数据量: {}, 耗时: {}ms, 总计: {}", 
                        reason, batch.size(), duration, newTotal);
                } else {
                    log.error("异步发送失败: {}", throwable.getMessage());
                    // 无锁重试：重新加入队列尾部（避免死循环）
                    int retryCount = 0;
                    for (NormalFieldsDTO data : batch) {
                        if (retryCount++ < batch.size() / 2) { // 只重试一半，避免积压
                            buffer.offer(data);
                        } else {
                            log.warn("放弃重试数据，避免积压");
                            break;
                        }
                    }
                }
            });
            
            log.debug("异步发送已提交 - 数据量: {}", batch.size());
                
        } catch (Exception e) {
            log.error("批量发送失败 - 数据量: {}, 错误: {}", batch.size(), e.getMessage(), e);
            
            // 无锁简单重试：部分重试避免积压
            int retryCount = 0;
            for (NormalFieldsDTO data : batch) {
                if (retryCount++ < 10) { // 最多重试10条
                    buffer.offer(data);
                } else {
                    break;
                }
            }
        }
    }
    
    /**
     * 定时刷新缓冲区（兜底机制）
     */
    @Scheduled(fixedDelayString = "${batch.timeout:2000}")
    public void scheduleFlush() {
        if (!buffer.isEmpty()) {
            long timeSinceLastSend = System.currentTimeMillis() - lastSendTime;
            if (timeSinceLastSend > batchTimeoutMs) {
                flushBuffer("定时刷新");
            }
        }
    }
    
    /**
     * 定时输出统计信息
     */
    @Scheduled(fixedDelay = 60000) // 每分钟输出一次
    public void logStats() {
        long sent = totalSent.get();
        long batches = totalBatches.get();
        if (sent > 0) {
            double avgBatchSize = batches > 0 ? (double) sent / batches : 0;
            log.info("Kafka发送统计 - 总发送: {}, 批次数: {}, 平均批次大小: {:.1f}, 当前缓冲: {}", 
                sent, batches, avgBatchSize, buffer.size());
        }
    }
    
    /**
     * 获取当前缓冲区状态
     */
    public String getBufferStatus() {
        return String.format("缓冲区: %d/%d, 已发送: %d 条 (%d 批次), Producer: %s", 
            buffer.size(), batchSize, totalSent.get(), totalBatches.get(), 
            producer != null ? producer.getStats() : "null");
    }
    
    /**
     * 获取当前缓冲区大小
     */
    public int getBufferSize() {
        return buffer.size();
    }
    
    /**
     * 手动清空缓冲区（慎用）
     */
    public void clearBuffer() {
        int size = buffer.size();
        buffer.clear();
        if (size > 0) {
            log.warn("手动清空缓冲区，丢弃 {} 条数据", size);
        }
    }
    
    /**
     * 应用关闭前刷新所有待发送数据
     */
    @PreDestroy
    public void shutdown() {
        log.info("应用关闭，刷新剩余数据...");
        
        // 多次刷新直到空
        int flushCount = 0;
        while (!buffer.isEmpty() && flushCount < 5) {
            flushBuffer("应用关闭");
            flushCount++;
            
            if (!buffer.isEmpty()) {
                try {
                    Thread.sleep(1000); // 等待异步发送完成
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    break;
                }
            }
        }
        
        // 关闭Producer
        if (producer != null) {
            producer.close();
        }
        
        if (!buffer.isEmpty()) {
            log.warn("应用关闭时仍有 {} 条数据未发送", buffer.size());
        }
        
        log.info("BatchKafkaSender关闭完成，总共发送: {} 条数据", totalSent.get());
    }
}