package com.zork.common.service;

import lombok.extern.slf4j.Slf4j;
import org.influxdb.InfluxDB;
import org.influxdb.InfluxDBFactory;
import org.influxdb.dto.Query;
import org.influxdb.dto.QueryResult;

@Slf4j
public class InfluxDBService {
    private InfluxDB influxDB;
    private String database;

    public InfluxDBService(String username, String password, String url, String database) {
        this.database = database;
        this.influxDB = InfluxDBFactory.connect(url, username, password);
    }

    public void influxDbBuild() {
        influxDB.setDatabase(database);
    }

    public void createRetentionPolicy(String policyName) {
        // 创建保留策略的逻辑
    }

    public QueryResult query(String command) {
        if (command == null || command.trim().isEmpty()) {
            throw new IllegalArgumentException("查询命令不能为空");
        }
        if (influxDB == null) {
            throw new IllegalStateException("InfluxDB连接未初始化");
        }
        
        log.info("执行InfluxQL查询: {}", command);
        long startTime = System.currentTimeMillis();
        try {
            QueryResult result = influxDB.query(new Query(command, database));
            long duration = System.currentTimeMillis() - startTime;
            log.info("查询执行完成，耗时: {}ms, 结果行数: {}", duration, getResultCount(result));
            return result;
        } catch (Exception e) {
            long duration = System.currentTimeMillis() - startTime;
            log.error("查询执行失败，耗时: {}ms, SQL: {}, 错误: {}", duration, command, e.getMessage());
            throw e;
        }
    }

    private int getResultCount(QueryResult result) {
        try {
            if (result != null && result.getResults() != null && !result.getResults().isEmpty()) {
                QueryResult.Result firstResult = result.getResults().get(0);
                if (firstResult.getSeries() != null && !firstResult.getSeries().isEmpty()) {
                    QueryResult.Series series = firstResult.getSeries().get(0);
                    return series.getValues() != null ? series.getValues().size() : 0;
                }
            }
            return 0;
        } catch (Exception e) {
            return -1;
        }
    }

    public void close() {
        if (influxDB != null) {
            influxDB.close();
        }
    }

    public static QueryResult.Series getResult(QueryResult queryResult) {
        if (queryResult != null && queryResult.getResults() != null 
            && !queryResult.getResults().isEmpty()
            && queryResult.getResults().get(0).getSeries() != null
            && !queryResult.getResults().get(0).getSeries().isEmpty()) {
            return queryResult.getResults().get(0).getSeries().get(0);
        }
        return null;
    }
}