package com.zork.htpocesb.thread;

import com.zork.common.service.InfluxDBService;
import com.zork.common.utils.DateUtil;
import lombok.extern.slf4j.Slf4j;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.function.BiConsumer;
import java.util.function.Function;

import static com.zork.htpocesb.component.CsvWriteInfluxDBComponent.invoke;

/**
 * @Author: Prock.Liy
 * @Date： 2021/7/26
 * @Description：
 */
@Slf4j
public class CompletableFutureAsync {

    /**
     * 此executor线程池如果不传,CompletableFuture经测试默认只启用最多3个线程,所以最好自己指定线程数量
     * <p>
     * 默认线程池定义，(15个)
     */
    protected static ExecutorService EXECUTOR_POOL = Executors.newFixedThreadPool(
            15
    );

    /**
     * Java8 CompletableFuture执行任务
     *
     * @param influxDBConnect influxDB链接
     * @param fileNameList    文件名
     */
    public static void execute(InfluxDBService influxDBConnect, String filePath, List<String> fileNameList) {
        long startTime = System.currentTimeMillis();

        CompletableFuture<Void> allFutures = CompletableFuture.allOf(
                fileNameList.stream()
                        .map(fileName -> handle(influxDBConnect, filePath, fileName))
                        .toArray(CompletableFuture[]::new)
        );

        allFutures.join();

        allFutures.whenComplete(new BiConsumer<Void, Throwable>() {
            @Override
            public void accept(Void aVoid, Throwable throwable) {
                if (allFutures.isDone()) {
                    log.info("Task execution success, time-consuming:【\033[31m{}\033[0m】", DateUtil.timeConsuming(startTime, System.currentTimeMillis()));
                }
                EXECUTOR_POOL.shutdown();
            }
        });
    }

    /**
     * 线程并行执行
     *
     * @param influxDBConnect
     * @param filePath
     * @param fileName
     * @return
     */
    public static CompletableFuture<Void> handle(InfluxDBService influxDBConnect, String filePath, String fileName) {
        return CompletableFuture.runAsync(() -> {
            String csvFilePath = String.format("%s\\%s", filePath, fileName);
            log.info("Start to read Csv file data, file path:  【\033[31m{}\033[0m】", csvFilePath);
            invoke(influxDBConnect, csvFilePath);
            try {
                Thread.sleep(3000);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }

        }, EXECUTOR_POOL).exceptionally(new Function<Throwable, Void>() { //捕捉异常,不会导致整个流程中断
            @Override
            public Void apply(Throwable throwable) {
                log.info("线程[{}]发生了异常, 继续执行其他线程,错误详情[{}]", Thread.currentThread().getName(), throwable.getMessage());
                return null;
            }
        });
    }

}
