package com.zork.gondb.component;

import com.alibaba.fastjson.JSONObject;
import com.zork.common.dto.Dimension;
import com.zork.common.dto.NormalFieldsDTO;
import com.zork.gondb.model.OrderDO;
import com.zork.gondb.model.OrderStatusDO;
import com.zork.gondb.service.IOrderService;
import com.zork.gondb.service.IOrderStatusService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.*;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.stream.Collectors;

/**
 * @Author: Prock.Liy
 * @Date： 2022/4/20
 * @Description： 查询gondb数据
 */
@Slf4j
@Component
public class GondbComponent {

    @Value("${send.log-url}")
    private String sendLogUrl;

    @Value("${minute}")
    private Integer minute;

    @Resource
    private RestTemplate restTemplate;

    @Resource
    private IOrderService iOrderService;

    @Resource
    private IOrderStatusService iOrderStatusService;

    /**
     * thread pools   如果线程池长度超过处理需要，可灵活回收空闲线程，若无可回收，则新建线程
     */
    protected ExecutorService EXECUTOR_POOL = Executors.newCachedThreadPool();

    /**
     * 执行Sql输出统计Json
     */
    @Scheduled(cron = "${schedules}")
    public void execute() {
        try {
            log.info("start query database......................");
            List<OrderDO> orderDOList = iOrderService.findOrderByTime(minute);
            List<OrderStatusDO> orderStatusDOList = iOrderStatusService.findOrderByTime(minute);
            EXECUTOR_POOL.execute(() -> {
                List<Long> orderStatusAgeIdS = new ArrayList<>();
                orderDOList.forEach(orderDO -> {
                    // 筛选相同agwSeqId的数据合并
                    OrderStatusDO orderStatusDO = orderStatusDOList.stream()
                            .filter(orderStatus -> orderDO.getAgwSeqId().equals(orderStatus.getAgwSeqId()))
                            .findAny().orElse(null);
                    if (!Objects.isNull(orderStatusDO)) {
                        orderDO.setOrdStatus(orderStatusDO.getOrdStatus());
                        orderDO.setRejectreasonCode(orderStatusDO.getRejectReasonCode());
                        orderDO.setOrdRejReason(orderStatusDO.getOrdRejReason());
                        orderDO.setExErrMsg(orderStatusDO.getExErrMsg());

                        // 用于判断是否存在agwSeqId不相同的操作
                        orderStatusAgeIdS.add(orderStatusDO.getAgwSeqId());
                    }


                    NormalFieldsDTO normalFieldsDTO = new NormalFieldsDTO();
                    normalFieldsDTO.setLogTypeName("dys_t_order");
                    normalFieldsDTO.setNormalFields(JSONObject.parseObject(JSONObject.toJSONString(orderDO), Map.class));
                    normalFieldsDTO.setDimensions(Dimension.builder()
                            .appsystem("dys").build());
                    normalFieldsDTO.setTimestamp(new Date().getTime());
                    // 发送Kafka
                    sendLog2Kafka(normalFieldsDTO);
                });

                // -------------------------遍历order_status表
//                orderStatusDOList.forEach(orderStatusDO -> {
//                    // agwSeqId存在未合并便写入Kafka
//                    boolean isExit = orderStatusAgeIdS.stream().anyMatch(agwSeqId -> agwSeqId.equals(orderStatusDO.getAgwSeqId()));
//                    if (!isExit) {
//                        NormalFieldsDTO normalFieldsDTO = new NormalFieldsDTO();
//                        normalFieldsDTO.setLogTypeName("dys_t_order_status");
//                        normalFieldsDTO.setNormalFields(JSONObject.parseObject(JSONObject.toJSONString(orderStatusDO), Map.class));
//                        normalFieldsDTO.setDimensions(Dimension.builder()
//                                .appsystem("dys").build());
//                        normalFieldsDTO.setTimestamp(new Date().getTime());
//                        // 发送Kafka
//                        sendLog2Kafka(normalFieldsDTO);
//                    }
//                });
            });
        } catch (Exception e) {
            log.error("\n Query Database Execute: -> {}", e.getMessage());
        }
    }


    /**
     * 发送指标数据至Kafka
     *
     * @param normalFieldsDTO
     * @return JSONObject
     */
    private void sendLog2Kafka(NormalFieldsDTO normalFieldsDTO) {
        try {
            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.APPLICATION_JSON);
            HttpEntity entity = new HttpEntity<>(JSONObject.toJSON(normalFieldsDTO), headers);
            ResponseEntity<String> result = restTemplate.exchange(sendLogUrl, HttpMethod.POST, entity, String.class);

            // 调用失败缺少measure值时，进行补齐重新调归补用递
            if (!Objects.requireNonNull(result.getBody()).contains("OK")) {
                log.info("sendLog2Kafka SUCCESS,调用参数: ->{}  ,接口调用Result: ->{}", JSONObject.toJSON(normalFieldsDTO), result.getBody());
            }
        } catch (Exception e) {
            log.error("sendLog2Kafka Exception: -> {} ,调用参数: ->{}", e.getMessage(), JSONObject.toJSON(normalFieldsDTO));
        }
    }

}
