package com.zork.disorder.component;


import com.zork.common.service.InfluxDBService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.concurrent.BasicThreadFactory;
import org.influxdb.dto.QueryResult;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Function;
import java.util.function.Predicate;

import static com.zork.common.constant.InfluxDBConstant.AUTOGEN;
import static com.zork.common.service.InfluxDBService.getResult;
import static com.zork.common.utils.DateUtil.yearMonthDayBar;

/**
 * @Author: Prock.Liy
 * @Date： 2021/6/1
 * @Description： 查询influxdb一分钟内数据
 */
@Slf4j
@Component("countComponent")
public class CountComponent {

    @Value("${influxdb.url}")
    private String urls;

    @Value("${influxdb.user}")
    private String user;

    @Value("${influxdb.password}")
    private String password;

    @Value("${influxdb.database}")
    private String database;

    @Value("${table-keywords}")
    private String tableKeywords;

    @Value("${spring.kafka.producer.topic}")
    private String topic;

    @Value("${spring.kafka.bootstrap-servers}")
    private String bootstrapServers;

    @Value("${interval-minute}")
    private Integer intervalMinute;


    @Value("${missing-s}")
    private Integer missingS;

    /**
     * 查询频率，分钟
     */
    @Value("${initialDelay}")
    private Integer initialDelay;


    private final SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");

    /**
     * thread pools   如果线程池长度超过处理需要，可灵活回收空闲线程，若无可回收，则新建线程
     */
    protected ExecutorService EXECUTOR_POOL = Executors.newCachedThreadPool();

    protected ScheduledExecutorService EXECUTOR_SERVICE = Executors.newScheduledThreadPool(1,
            new BasicThreadFactory.Builder()
                    .namingPattern("EXAMPLE-SCHEDULE-POOL-%d")
                    .daemon(true).build());


    /**
     * 查询influxdb指标数据发送至Kafka
     */
//    @Scheduled(cron = "${schedules}")
    public void execute() {
        try {
            // 截取influxdb地址
            List<String> urlList = Arrays.asList(urls.split(","));
            String startTime = simpleDateFormat.format(lastFewMinutesMissingS(intervalMinute));
            String endTime = simpleDateFormat.format(new Date());
            EXECUTOR_POOL.execute(() -> {
                try {
                    urlList.forEach(url -> {
                        // 创建连接
                        InfluxDBService influxDBService = new InfluxDBService(user, password, url, database);
                        influxDBService.influxDbBuild();
                        influxDBService.createRetentionPolicy(AUTOGEN);

                        // 获取今日需要查询的所有表名
                        List<String> tableNameList = getTableNameList(influxDBService);

                        if (tableNameList.size() == 0) {
                            return;
                        }

                        Map<String, String> tableCount = new HashMap<>();
                        // 查询数据写入Kafka
                        tableNameList.forEach(tableName -> {
                            String sql = String.format("select count(*) from \"%s\" ",
                                    tableName, startTime, endTime);
                            QueryResult.Series series = getResult(influxDBService.query(sql));

                            // 获取今日所有表名
                            if (series == null) {
                                return;
                            }

                            tableCount.put(simpleDateFormat.format(new Date()), series.getValues().get(0).get(1).toString());
                            // 统计每张表的count（仅记录日志）
                            log.info("表 {} 统计结果: {}", tableName, tableCount.toString());
                        });
                        // 更新count文件
                        influxDBService.close();
                    });

                } catch (Exception e) {
                    log.error("query influx Exception:{}", e.getMessage());
                }
            });
        } catch (Exception e) {
            log.error("task Exception:{}", e.getMessage());
        }
    }

    /**
     * 获取过去几分钟时间,因为数据缺失，要求减去3秒
     *
     * @param minutes 分钟
     * @return
     */
    public Date lastFewMinutesMissingS(int minutes) {
        Calendar beforeTime = Calendar.getInstance();
        beforeTime.add(Calendar.MINUTE, -minutes);
        beforeTime.add(Calendar.SECOND, -missingS);
        return beforeTime.getTime();
    }

    /**
     * 根据特定值去重
     *
     * @param keyExtractor
     * @param <T>
     * @return
     */
    public static <T> Predicate<T> distinctByKey(Function<? super T, Object> keyExtractor) {
        Map<Object, Boolean> map = new ConcurrentHashMap<>();
        return t -> map.putIfAbsent(keyExtractor.apply(t), Boolean.TRUE) == null;
    }

    /**
     * 获取今日所有表名
     *
     * @param influxDBService 连接池
     * @return JhAppHqOnLine
     */
    public List<String> getTableNameList(InfluxDBService influxDBService) {
        List<String> tableNameList = new ArrayList<>();
        try {
            // 查询所有表名，筛选出需查询的表明
            QueryResult result = influxDBService.query("show measurements");
            QueryResult.Series series = getResult(result);

            // 获取今日所有表名
            assert series != null;
            series.getValues().forEach(value -> {
                if (value.get(0).toString().contains(tableKeywords) && value.get(0).toString().contains("credit") && value.get(0).toString().contains(yearMonthDayBar())) {
                    tableNameList.add(value.get(0).toString());
                }
            });
        } catch (Exception e) {
            log.error("InfluxDBDataConversion Exception:{}", e.getMessage());
        }
        return tableNameList;
    }


}
