package com.zork.rateinfo.component;

import com.alibaba.fastjson.JSONObject;
import com.zork.common.dto.MetricDTO;
import com.zork.common.service.InfluxDBService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.influxdb.dto.QueryResult;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.*;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import static com.zork.common.constant.DimensionConstant.APP_SYSTEM;
import static com.zork.common.constant.InfluxDBConstant.AUTOGEN;
import static com.zork.common.service.InfluxDBService.getResult;
import static com.zork.common.utils.DateUtil.parseSSSTime;

/**
 * @Author: Prock.Liy
 * @Date： 2022/4/20
 * @Description： 查询es下当天所有指标的count条数以及大小指标
 */
@Slf4j
@Component
public class RateInfoComponent {

    @Value("${send.url}")
    private String sendUrl;

    @Value("${interval-seconds}")
    private Integer intervalSeconds;

    @Value("${metricsetname}")
    private String metricsetname;

    @Value("${influxdb.url}")
    private String urls;

    @Value("${influxdb.user}")
    private String user;

    @Value("${influxdb.password}")
    private String password;

    @Value("${influxdb.database}")
    private String database;

    @Resource
    private RestTemplate restTemplate;

    /**
     * thread pools   如果线程池长度超过处理需要，可灵活回收空闲线程，若无可回收，则新建线程
     */
    protected ExecutorService EXECUTOR_POOL = Executors.newCachedThreadPool();

    /**
     *
     */
    @Scheduled(cron = "${schedules}")
    public void execute() {
        try {
            // 截取influxdb地址
            List<String> urlList = Arrays.asList(urls.split(","));
            EXECUTOR_POOL.execute(() -> {
                try {
                    urlList.forEach(url -> {
                        // 创建连接
                        InfluxDBService influxDBService = new InfluxDBService(user, password, url, database);
                        influxDBService.influxDbBuild();
                        influxDBService.createRetentionPolicy(AUTOGEN);

                        String sql = String.format("SELECT * FROM \"RateInfo\" WHERE time > now() - %ss", intervalSeconds);
                        QueryResult.Series series = getResult(influxDBService.query(sql));

                        // 获取今日所有表名
                        if (series == null) {
                            return;
                        }

                        series.getValues().forEach(value -> {
                            Map<String, String> dimensions = new HashMap<>();
                            Map<String, String> measures = new HashMap<>();

                            dimensions.put(APP_SYSTEM, "DYSJZJYSystem");
                            dimensions.put("ip",StringUtils.substringAfterLast(StringUtils.substringBeforeLast(url,":"),"/"));
                            dimensions.put("CheckPoint", value.get(1).toString());

                            measures.put("rate", value.get(5).toString());

                            MetricDTO metricDTO = new MetricDTO();
                            metricDTO.setMetricsetname(metricsetname);
                            metricDTO.setDimensions(dimensions);
                            metricDTO.setMeasures(measures);
                            metricDTO.setTimestamp(parseSSSTime(StringUtils.substringBefore(value.get(0).toString(),"."),"yyyy-MM-dd'T'HH:mm:ss"));

                            // 发送数据至kafka
                            sendMetric2Kafka(metricDTO);
                        });
                        influxDBService.close();
                    });

                } catch (Exception e) {
                    log.error("query influx Exception:{}", e.getMessage());
                }
            });
        } catch (Exception e) {
            log.error("\n RateInfo Exception: -> {}", e.getMessage());
        }
    }

    /**
     * 发送指标数据至Kafka
     *
     * @param metricDTO
     * @return JSONObject
     */
    private void sendMetric2Kafka(MetricDTO metricDTO) {
        try {
            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.APPLICATION_JSON);
            HttpEntity entity = new HttpEntity<>(JSONObject.toJSON(metricDTO), headers);
            ResponseEntity<String> result = restTemplate.exchange(sendUrl, HttpMethod.POST, entity, String.class);

            // 调用失败缺少measure值时，进行补齐重新调归补用递
            if (!Objects.requireNonNull(result.getBody()).contains("OK")) {
                log.info("sendMetric2Kafka SUCCESS,调用参数: ->{}  ,接口调用Result: ->{}", JSONObject.toJSON(metricDTO), result.getBody());
            }
        } catch (Exception e) {
            log.error("sendMetric2Kafka Exception: -> {} ,调用参数: ->{}", e.getMessage(), JSONObject.toJSON(metricDTO));
        }
    }


}
