package com.zork.parse.component;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.zork.common.dto.MetricDTO;
import com.zork.common.utils.RestTemplateUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.*;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import static com.zork.common.constant.DimensionConstant.*;
import static com.zork.common.utils.DateUtil.yearMonthDayPoint;
import static com.zork.parse.constant.IndicesConstant.COUNT;
import static com.zork.parse.constant.IndicesConstant.STORE_SIZE;

/**
 * @Author: Prock.Liy
 * @Date： 2022/4/20
 * @Description： 查询es下当天所有指标的count条数以及大小指标
 */
@Slf4j
@Component
public class EsIndicesComponent {

    @Value("${send.metric-url}")
    private String sendMetricUrl;

    @Value("${es.indices-url}")
    private String indicesUrl;

    @Value("${es.count-url}")
    private String countUrl;

    @Value("${es.count-sql}")
    private String countSql;

    @Resource
    private RestTemplate restTemplate;

    /**
     * thread pools   如果线程池长度超过处理需要，可灵活回收空闲线程，若无可回收，则新建线程
     */
    protected ExecutorService EXECUTOR_POOL = Executors.newCachedThreadPool();

    /**
     * 执行Sql输出统计Json
     */
    @Scheduled(cron = "${schedules}")
    public void execute() {
        try {
            // 获取当天所有索引名称以及storeSize
            JSONArray indicesJsonArray = RestTemplateUtil.getJSONArray(String.format(indicesUrl, yearMonthDayPoint()));
            // 构建指标值
            EXECUTOR_POOL.execute(() -> {
                for (Object object : indicesJsonArray) {
                    // 获取index每日storeSize指标
                    JSONObject indicesObject = JSONObject.parseObject(JSON.toJSONString(object));

                    Map<String, String> dimensions = new HashMap<>();
                    String index = indicesObject.getString(INDEX);
                    dimensions.put(INDEX, index);
                    dimensions.put(APP_SYSTEM, "ZorkSystem");

                    MetricDTO metricDTO = new MetricDTO();
                    metricDTO.setMetricsetname("indices");
                    metricDTO.setDimensions(dimensions);
                    metricDTO.setTimestamp(new Date().getTime());
                    // 构建measures值
                    if (indicesObject.getString("store.size").contains("gb")) {
                        metricDTO.setMeasures(buildMeasuresValue(STORE_SIZE, StringUtils.substringBefore(indicesObject.getString("store.size"), "gb")));
                        // 发送数据至kafka
                        sendMetric2Kafka(metricDTO);
                    }
                    // 根据index查群count指标
                    getIndexCount(index);
                }
            });
        } catch (Exception e) {
            log.error("\n Query IndicesExecute: -> {}", e.getMessage());
        }
    }

    /**
     * 获取index下logTypeName的count条数
     *
     * @param index
     */
    private void getIndexCount(String index) {
        // 获取当天所有索引名称以及storeSize
        String esUrl = String.format(countUrl, index);
        log.info("start query,esUrl:{}", esUrl);
        JSONObject jsonObject = RestTemplateUtil.postJSONObject(esUrl, JSONObject.parseObject(countSql));

        if (jsonObject.isEmpty()) {
            log.info("no Data");
            return;
        }
        // 获取数据数组
        JSONArray buckets = jsonObject.getJSONObject("aggregations")
                .getJSONObject("dimensions.appsystem.keyword")
                .getJSONArray("buckets");

        // 构建指标值
        for (Object object : buckets) {
            // 获取index每日storeSize指标
            JSONObject indicesObject = JSONObject.parseObject(JSON.toJSONString(object));

            String appSystem = indicesObject.getString("key");
            JSONArray bucketsArray = indicesObject.getJSONObject("logTypeName.keyword").getJSONArray("buckets");

            Map<String, String> dimensions = new HashMap<>();
            dimensions.put(APP_SYSTEM, appSystem);
            // 获取系统下不同logTypeName的count指标
            for (Object valueObject : bucketsArray) {
                // 获取index每日storeSize指标
                JSONObject valueJson = JSONObject.parseObject(JSON.toJSONString(valueObject));

                dimensions.put(LOG_TYPE_NAME, valueJson.getString("key"));

                MetricDTO metricDTO = new MetricDTO();
                metricDTO.setMetricsetname("index_day_count");
                metricDTO.setDimensions(dimensions);
                metricDTO.setTimestamp(new Date().getTime());
                // 构建measures值
                metricDTO.setMeasures(buildMeasuresValue(COUNT, valueJson.getString("doc_count")));
                // 发送数据至kafka
                sendMetric2Kafka(metricDTO);
            }
        }
    }

    /**
     * @param size
     * @return measures
     */
    private Map<String, String> buildMeasuresValue(String metricsetName, String size) {
        Map<String, String> measures = new HashMap<>(1);
        measures.put(metricsetName, size);
        return measures;
    }

    /**
     * 发送指标数据至Kafka
     *
     * @param metricDTO
     * @return JSONObject
     */
    private void sendMetric2Kafka(MetricDTO metricDTO) {
        try {
            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.APPLICATION_JSON);
            HttpEntity entity = new HttpEntity<>(JSONObject.toJSON(metricDTO), headers);
            ResponseEntity<String> result = restTemplate.exchange(sendMetricUrl, HttpMethod.POST, entity, String.class);

            // 调用失败缺少measure值时，进行补齐重新调归补用递
            if (!Objects.requireNonNull(result.getBody()).contains("OK")) {
                log.info("sendMetric2Kafka SUCCESS,调用参数: ->{}  ,接口调用Result: ->{}", JSONObject.toJSON(metricDTO), result.getBody());
            }
        } catch (Exception e) {
            log.error("sendMetric2Kafka Exception: -> {} ,调用参数: ->{}", e.getMessage(), JSONObject.toJSON(metricDTO));
        }
    }

}
