package com.zork.parse.kafka;

import com.alibaba.fastjson.JSONObject;
import com.gtja.service.ParseLogService;
import com.zork.common.dto.Dimension;
import com.zork.common.dto.NormalFieldsDTO;
import com.zork.parse.model.ParseDTO;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.kafka.clients.consumer.ConsumerConfig;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.springframework.http.*;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.io.UnsupportedEncodingException;
import java.util.*;

import static com.zork.parse.constant.ParseConstant.*;

/**
 * @Author: Prock.Liy
 * @Date： 2021/8/10
 * @Description：
 */
@Slf4j
//@Component
public class Consumer {

    KafkaConsumer<String, byte[]> consumer;

    private final String topic;

    private final String logTypeName;

    private int count;

    public Consumer(String topic, String servers, String groupId, String reset, String logTypeName) {
        Properties props = new Properties();
        props.put(ConsumerConfig.BOOTSTRAP_SERVERS_CONFIG, servers);
        props.put(ConsumerConfig.GROUP_ID_CONFIG, groupId);
        props.put(ConsumerConfig.ENABLE_AUTO_COMMIT_CONFIG, "true");
        props.put(ConsumerConfig.AUTO_COMMIT_INTERVAL_MS_CONFIG, "1000"); //自动提交(批量确认)
        props.put(ConsumerConfig.MAX_POLL_RECORDS_CONFIG, 1000);
        props.put(ConsumerConfig.SESSION_TIMEOUT_MS_CONFIG, "30000");
        props.put(ConsumerConfig.AUTO_OFFSET_RESET_CONFIG, reset); //这个属性. 它能够消费昨天发布的数据
        props.put(ConsumerConfig.KEY_DESERIALIZER_CLASS_CONFIG, org.apache.kafka.common.serialization.ByteArrayDeserializer.class);
        props.put(ConsumerConfig.VALUE_DESERIALIZER_CLASS_CONFIG, org.apache.kafka.common.serialization.ByteArrayDeserializer.class);
        this.consumer = new KafkaConsumer<String, byte[]>(props);
        this.topic = topic;
        this.logTypeName = logTypeName;
        this.consumer.subscribe(Arrays.asList(topic));
    }

    public void messageConsumer(RestTemplate restTemplate, String sendLogUrl) throws InterruptedException {
        consumer.subscribe(Collections.singleton(this.topic));
        while (true) {
            ConsumerRecords<String, byte[]> consumerRecords = consumer.poll(100);
            if (consumerRecords.count() == 0 || consumerRecords.isEmpty()) {
                log.info("Kafka data not obtained....");
                continue;
            }
            Thread.sleep(1000);
            // 数据信息解密
            consumerRecords.forEach((ConsumerRecord<String, byte[]> record) -> {
                try {
                    count++;
                    log.info("count:  {}", count);
                    ParseLogService parseLogService = new ParseLogService();
                    String message = parseLogService.parse(record.value());
                    //log.info("message:  {}", message);
                    if (StringUtils.isEmpty(message) || !message.contains(CALL_STR)) {
                        log.info("Message is Null...");
                        return;
                    }

                    NormalFieldsDTO normalFieldsDTO = new NormalFieldsDTO();
                    normalFieldsDTO.setLogTypeName(logTypeName);
                    JSONObject jsonObject = JSONObject.parseObject(message);
                    if (jsonObject.isEmpty()) {
                        return;
                    }
                    ParseDTO parseDTO = getJsonValue(jsonObject);
                    if (Objects.isNull(parseDTO)) {
                        return;
                    }
                    Map<String, Double> measures = new HashMap<>(1);
                    measures.put(SR_SS_TIME, parseDTO.getSR_SS_TIME());

                    Map<String, String> normalFields = new HashMap<>(10);
                    normalFields.put(CALL_STR, parseDTO.getCALL_STR());
                    normalFields.put(DST_INFO, parseDTO.getDST_INFO());
                    normalFields.put(ERROR_INFO, parseDTO.getERROR_INFO());
                    normalFields.put(ERROR_NO, parseDTO.getERROR_NO());
                    normalFields.put(REQUEST_SUCCESS, parseDTO.getREQUEST_SUCCESS());
                    normalFields.put(SERVER_CLUSTER_NAME, parseDTO.getSERVER_CLUSTER_NAME());
                    normalFields.put(SERVER_IP, parseDTO.getSERVER_IP());
                    normalFields.put(TRACE_ID, parseDTO.getTRACE_ID());
                    normalFields.put(TIMESTAMP,parseDTO.getTIMESTAMP());

                    normalFieldsDTO.setSource("Kafka");
                    normalFieldsDTO.setOffset("1");
                    normalFieldsDTO.setMeasures(measures);
                    normalFieldsDTO.setNormalFields(normalFields);
                    normalFieldsDTO.setDimensions(Dimension.builder()
                            .appsystem("ZHLCLog").build());
                    normalFieldsDTO.setTimestamp(new Date().getTime());

                    // 发送至Kafka
                    try {
                        HttpHeaders headers = new HttpHeaders();
                        headers.setContentType(MediaType.APPLICATION_JSON);
                        HttpEntity entity = new HttpEntity<>(JSONObject.toJSON(normalFieldsDTO), headers);
                        ResponseEntity<String> result = restTemplate.exchange(sendLogUrl, HttpMethod.POST, entity, String.class);

                        // 调用失败缺少measure值时，进行补齐重新调归补用递
                        if (!Objects.requireNonNull(result.getBody()).contains("OK")) {
                            log.info("sendMetric2Kafka SUCCESS,调用参数: ->{}  ,接口调用Result: ->{}", JSONObject.toJSON(normalFieldsDTO), result.getBody());
                        }
                    } catch (Exception e) {
                        log.error("sendMetric2Kafka Exception: -> {} ,调用参数: ->{}", e.getMessage(), JSONObject.toJSON(normalFieldsDTO));
                    }
                } catch (UnsupportedEncodingException e) {
                    log.error("messageConsumer Exception: -> {}", e.getMessage());
                }
            });
        }
    }


    /**
     * 获取Json对象值
     *
     * @param jsonObject
     * @return
     */
    public ParseDTO getJsonValue(JSONObject jsonObject) {
        try {
            return ParseDTO.builder()
                    .TIMESTAMP(jsonObject.containsKey(TIMESTAMP) ? jsonObject.getString(TIMESTAMP) : StringUtils.EMPTY)
                    .CALL_STR(jsonObject.containsKey(CALL_STR) ? jsonObject.getString(CALL_STR) : StringUtils.EMPTY)
                    .DST_INFO(jsonObject.containsKey(DST_INFO) ? jsonObject.getJSONArray(DST_INFO).toString() : StringUtils.EMPTY)
                    .ERROR_INFO(jsonObject.containsKey(ERROR_INFO) ? jsonObject.getString(ERROR_INFO) : StringUtils.EMPTY)
                    .ERROR_NO(jsonObject.containsKey(ERROR_NO) ? jsonObject.getString(ERROR_NO) : StringUtils.EMPTY)
                    .REQUEST_SUCCESS(jsonObject.containsKey(REQUEST_SUCCESS) ? jsonObject.getString(REQUEST_SUCCESS) : StringUtils.EMPTY)
                    .SERVER_CLUSTER_NAME(jsonObject.containsKey(SERVER_CLUSTER_NAME) ? jsonObject.getString(SERVER_CLUSTER_NAME) : StringUtils.EMPTY)
                    .SERVER_IP(jsonObject.containsKey(SERVER_IP) ? jsonObject.getString(SERVER_IP) : StringUtils.EMPTY)
                    .SR_SS_TIME(jsonObject.containsKey(SR_SS_TIME) ? jsonObject.getDouble(SR_SS_TIME) : 0.0)
                    .TRACE_ID(jsonObject.containsKey(TRACE_ID) ? jsonObject.getString(TRACE_ID) : StringUtils.EMPTY)
                    .build();
        } catch (Exception e) {
            log.error("getJsonValue Exception: -> {} ,调用参数: ->{}", e.getMessage(), JSONObject.toJSON(jsonObject));
        }
        return new ParseDTO();
    }

//    public static void main(String[] args) throws InterruptedException {
//        new Consumer("apm", "10.180.61.181:9092", "qwerqwer10086", "latest", "test").messageConsumer();
//    }

}
