package com.zork.commission.component;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.zork.common.constant.DateConstant;
import com.zork.commission.dto.DataSourceDTO;
import com.zork.common.dto.ExecuteResultDTO;
import com.zork.commission.properties.DataSourceProperties;
import com.zork.common.utils.SqlFileExecuteUtil;
import com.zork.common.utils.StringUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.*;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import java.text.SimpleDateFormat;
import java.util.*;

import static com.zork.common.utils.FileUtil.readerFileRow;

/**
 * @Author: Prock.Liy
 * @Date： 2021/6/1
 * @Description： 数据检测组件
 */
@Slf4j
@Component
public class CommissionComponent {

    @Resource
    private DataSourceProperties dataSourceProperties;

    @Resource
    private RestTemplate restTemplate;

    @Value("${logstash.post-url}")
    private String logstashUrl;

    /**
     * cron调用
     */
    @Scheduled(cron = "${schedules}")
    public void execute() {
        try {
            dataSourceProperties.getDataSourceDTOList().forEach(this::invoke);
        } catch (Exception e) {
            log.error("  SqlQueryException: -> {}", e.getMessage());
        }
    }


    /**
     * 具体业务
     *
     * @param dataSourceDTO 数据源
     */
    public void invoke(DataSourceDTO dataSourceDTO) {
        try {
            // 执行Sql文件
            ExecuteResultDTO executeResultDTO = SqlFileExecuteUtil.executeSql("11",
                    dataSourceDTO.getUrl(),
                    dataSourceDTO.getUsername(),
                    dataSourceDTO.getPassword(),
                    dataSourceDTO.getFilePath());
            if (executeResultDTO.getStatus() == 1) {
                return;
            }

            // 解析数据返回结果-获取列名称与数据
            String sqlResult = Objects.requireNonNull(StringUtil.stringIntercept(executeResultDTO.getLog(), readerFileRow(dataSourceDTO.getFilePath()))).trim();

            List<String> columnList = Arrays.asList(StringUtils.substringBefore(sqlResult, "\n").split("\t"));
            List<String> valueList = Arrays.asList(StringUtils.substringAfter(sqlResult, "\n").split("\t"));

            boolean valueIs = valueList.stream().allMatch(StringUtils::isEmpty);
            if (valueIs) {
                log.info("  当前Sql查询无数据!");
                return;
            }

            // 转换为日志集格式
            dataConversionJson(columnList, valueList, dataSourceDTO.getHostname());
        } catch (Exception e) {
            e.printStackTrace();
            log.error("  invokeException: -> {}", e.getMessage());
        }

    }

    /**
     * Json格式转换
     *
     * @param columnList 列list
     * @param valueList  值list
     * @param hostname   主机名
     * @return JSONObject
     */
    private JSONObject dataConversionJson(List<String> columnList, List<String> valueList, String hostname) {
        JSONObject jsonObject = new JSONObject();
        try {
            JSONArray normalFieldsArray = new JSONArray();
            Map<String, String> normalFields = new HashMap<>(columnList.size());

            // jsonArray大小未等于数据库查询条数便一直循环
            int k = 0;
            for (int i = 0; normalFieldsArray.size() != (valueList.size() / columnList.size()); i++) {
                if (i == columnList.size()) {
                    i = 0;
                }
                if (k < valueList.size()) {
                    normalFields.put(columnList.get(i), valueList.get(k).replaceAll("\n", ""));
                    k++;
                }
                if (normalFields.size() == columnList.size()) {
                    normalFieldsArray.add(normalFields);
                    // 第二条开始初始化HashMap
                    normalFields = new HashMap<>(columnList.size());
                }
            }
            Map<String, String> dimensionHash = new HashMap<>(3);
            dimensionHash.put("appsystem", "jzjy");
            dimensionHash.put("hostname", hostname);
            dimensionHash.put("appprogramname", "sqlServer");

            // before 8 hour
            Calendar cal = Calendar.getInstance();
            cal.setTime(new Date());
            cal.add(Calendar.HOUR_OF_DAY, -8);
            jsonObject.put("timestamp", new SimpleDateFormat(DateConstant.STANDARD_DATE_FORMAT_UTC).format(cal.getTime()));
            jsonObject.put("dimensions", dimensionHash);
            jsonObject.put("logTypeName", "commission_sqlServer_log");
            if (normalFieldsArray.size() > 0) {
                normalFieldsArray.forEach(message -> {
                    jsonObject.put("normalFields", message);
                    restTemplateExchange(jsonObject);
                });
            }

        } catch (Exception e) {
            e.printStackTrace();
            log.error("  JSON转换异常: -> {}", e.getMessage());
        }
        return jsonObject;
    }

    /**
     * 发送数据至kafuka日志集topic
     *
     * @param requestParam
     */
    public void restTemplateExchange(JSONObject requestParam) {
        try {
            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.APPLICATION_JSON);
            HttpEntity entity = new HttpEntity<>(requestParam, headers);
            ResponseEntity<String> result = restTemplate.exchange(logstashUrl, HttpMethod.POST, entity, String.class);
            log.info("转换JSON格式后: ->{}  ,Logstash接口调用Result: ->{}", requestParam, result.getBody());
        } catch (Exception e) {
            e.printStackTrace();
            log.error(" 调用log接口异常,请求参数为: {},Exception: {}", requestParam, e.getMessage());
        }
    }

}
