package com.zork.disorder.kafka;

import com.alibaba.fastjson.JSONObject;
import com.zork.common.dto.NormalFieldsDTO;
import lombok.extern.slf4j.Slf4j;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.ProducerConfig;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.common.serialization.IntegerSerializer;
import org.apache.kafka.common.serialization.StringSerializer;

import java.util.List;
import java.util.Properties;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicLong;

/**
 * @Author: Prock.Liy
 * @Date： 2021/8/10
 * @Description：
 */
@Slf4j
public class Producer {

    private final KafkaProducer<Integer, String> producer;
    private final String topic;

    public Producer(KafkaSendDTO kafkaSendDTO) {
        Properties properties = new Properties();
        properties.put(ProducerConfig.BOOTSTRAP_SERVERS_CONFIG, kafkaSendDTO.getBootstrapServers());
        properties.put(ProducerConfig.KEY_SERIALIZER_CLASS_CONFIG, IntegerSerializer.class.getName());
        properties.put(ProducerConfig.VALUE_SERIALIZER_CLASS_CONFIG, StringSerializer.class.getName());
        
        // 高性能配置：异步批量发送
        properties.put(ProducerConfig.BATCH_SIZE_CONFIG, 65536);        // 64KB批次
        properties.put(ProducerConfig.LINGER_MS_CONFIG, 10);           // 10ms等待批次
        properties.put(ProducerConfig.BUFFER_MEMORY_CONFIG, 67108864); // 64MB缓冲
        properties.put(ProducerConfig.COMPRESSION_TYPE_CONFIG, "snappy"); // 压缩
        properties.put(ProducerConfig.ACKS_CONFIG, "1");               // 快速确认
        properties.put(ProducerConfig.RETRIES_CONFIG, 3);
        properties.put(ProducerConfig.MAX_IN_FLIGHT_REQUESTS_PER_CONNECTION, 5);

        producer = new KafkaProducer<Integer, String>(properties);
        this.topic = kafkaSendDTO.getTopic();
    }

    private final AtomicLong successCount = new AtomicLong(0);
    private final AtomicLong errorCount = new AtomicLong(0);

    /**
     * 异步批量发送消息至Kafka - 无阻塞
     */
    public CompletableFuture<Void> sendMessage(List<NormalFieldsDTO> normalFieldsDTOList, String countFile) {
        if (normalFieldsDTOList == null || normalFieldsDTOList.isEmpty()) {
            return CompletableFuture.completedFuture(null);
        }

        return CompletableFuture.runAsync(() -> {
            try {
                // 异步发送所有消息，无等待
                normalFieldsDTOList.forEach(normalFieldsDTO -> {
                    String message = JSONObject.toJSONString(normalFieldsDTO);
                    producer.send(new ProducerRecord<>(topic, message), (metadata, exception) -> {
                        if (exception == null) {
                            successCount.incrementAndGet();
                            if (log.isDebugEnabled()) {
                                log.debug("Kafka sent - offset:{}, partition:{}", 
                                    metadata.offset(), metadata.partition());
                            }
                        } else {
                            errorCount.incrementAndGet();
                            log.error("Kafka send failed: {}", exception.getMessage());
                        }
                    });
                });
                
                // 立即刷新，确保发送
                producer.flush();
                
                log.info("批量发送完成: {} 条消息, 成功: {}, 失败: {}", 
                    normalFieldsDTOList.size(), successCount.get(), errorCount.get());
                    
            } catch (Exception e) {
                log.error("批量发送异常: {}", e.getMessage(), e);
                throw new RuntimeException(e);
            }
        });
    }

    /**
     * 获取发送统计
     */
    public String getStats() {
        return String.format("成功: %d, 失败: %d", successCount.get(), errorCount.get());
    }

    /**
     * 关闭producer
     */
    public void close() {
        if (producer != null) {
            producer.close();
        }
    }

//    public static void main(String[] args) {
//        String countFile = "E:\\code\\applets\\zork-server\\influx-disorder\\src\\main\\resources\\count.txt";
//        String count = readFile(countFile);
//        String nowCount = String.valueOf((Integer.parseInt(count.replaceAll("\n","")) + 12));
//        appendMethodB(countFile, nowCount, false);
//        System.out.println(readFile(countFile));
//
//    }

}
