package com.zork.disorder.component;

import com.zork.common.service.InfluxDBService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import static com.zork.common.constant.InfluxDBConstant.AUTOGEN;

/**
 * @Author: Optimized
 * @Date: 2024
 * @Description: 多库连接管理器，复用InfluxDB连接，支持多URL多数据库
 */
@Slf4j
@Component
public class MultiDbConnectionManager {
    
    @Value("${influxdb.url}")
    private String urls;
    
    @Value("${influxdb.user}")
    private String user;
    
    @Value("${influxdb.password}")
    private String password;
    
    @Value("${influxdb.database}")
    private String databases;
    
    // 连接缓存池：key=url|database, value=InfluxDBService
    private final Map<String, InfluxDBService> connectionPool = new ConcurrentHashMap<>();
    
    // 健康状态缓存
    private final Map<String, Boolean> healthStatus = new ConcurrentHashMap<>();
    
    /**
     * 初始化所有连接
     */
    @PostConstruct
    public void initializeConnections() {
        List<String> urlList = Arrays.asList(urls.split(","));
        List<String> databaseList = Arrays.asList(databases.split(","));
        
        log.info("开始初始化InfluxDB连接，URL数量: {}, 数据库数量: {}", urlList.size(), databaseList.size());
        
        for (String url : urlList) {
            for (String database : databaseList) {
                try {
                    createConnection(url.trim(), database.trim());
                } catch (Exception e) {
                    log.error("初始化连接失败 - URL: {}, Database: {}, 错误: {}", 
                        url, database, e.getMessage());
                }
            }
        }
        
        log.info("InfluxDB连接池初始化完成，成功连接数量: {}", connectionPool.size());
    }
    
    /**
     * 获取指定URL和数据库的连接
     * 
     * @param url InfluxDB地址
     * @param database 数据库名
     * @return InfluxDB连接服务
     */
    public InfluxDBService getConnection(String url, String database) {
        String connectionKey = buildConnectionKey(url, database);
        
        // 先从缓存获取
        InfluxDBService service = connectionPool.get(connectionKey);
        
        if (service == null) {
            // 缓存中没有，创建新连接
            synchronized (this) {
                service = connectionPool.get(connectionKey);
                if (service == null) {
                    service = createConnection(url, database);
                }
            }
        }
        
        // 检查连接健康状态
        if (!isConnectionHealthy(connectionKey, service)) {
            log.warn("连接不健康，尝试重新创建 - URL: {}, Database: {}", url, database);
            service = recreateConnection(url, database);
        }
        
        return service;
    }
    
    /**
     * 创建新连接
     */
    private InfluxDBService createConnection(String url, String database) {
        String connectionKey = buildConnectionKey(url, database);
        
        try {
            log.debug("创建InfluxDB连接 - URL: {}, Database: {}", url, database);
            
            InfluxDBService service = new InfluxDBService(user, password, url, database);
            service.influxDbBuild();
            service.createRetentionPolicy(AUTOGEN);
            
            // 测试连接
            service.query("show measurements limit 1");
            
            connectionPool.put(connectionKey, service);
            healthStatus.put(connectionKey, true);
            
            log.info("成功创建InfluxDB连接 - URL: {}, Database: {}", url, database);
            return service;
            
        } catch (Exception e) {
            healthStatus.put(connectionKey, false);
            log.error("创建InfluxDB连接失败 - URL: {}, Database: {}, 错误: {}", 
                url, database, e.getMessage());
            throw new RuntimeException("无法连接到InfluxDB: " + url + "/" + database, e);
        }
    }
    
    /**
     * 重新创建连接
     */
    private InfluxDBService recreateConnection(String url, String database) {
        String connectionKey = buildConnectionKey(url, database);
        
        // 关闭旧连接
        InfluxDBService oldService = connectionPool.remove(connectionKey);
        if (oldService != null) {
            try {
                oldService.close();
            } catch (Exception e) {
                log.warn("关闭旧连接时出错: {}", e.getMessage());
            }
        }
        
        // 创建新连接
        return createConnection(url, database);
    }
    
    /**
     * 检查连接健康状态
     */
    private boolean isConnectionHealthy(String connectionKey, InfluxDBService service) {
        if (service == null) {
            return false;
        }
        
        Boolean lastStatus = healthStatus.get(connectionKey);
        if (lastStatus != null && !lastStatus) {
            return false; // 之前检查不健康
        }
        
        try {
            // 简单的健康检查查询
            service.query("show measurements limit 1");
            healthStatus.put(connectionKey, true);
            return true;
        } catch (Exception e) {
            log.warn("连接健康检查失败 - {}: {}", connectionKey, e.getMessage());
            healthStatus.put(connectionKey, false);
            return false;
        }
    }
    
    /**
     * 构建连接键：url|database
     */
    private String buildConnectionKey(String url, String database) {
        return url + "|" + database;
    }
    
    /**
     * 获取所有可用的URL列表
     */
    public List<String> getAvailableUrls() {
        return Arrays.asList(urls.split(","));
    }
    
    /**
     * 获取所有可用的数据库列表
     */
    public List<String> getAvailableDatabases() {
        return Arrays.asList(databases.split(","));
    }
    
    /**
     * 获取连接池统计信息
     */
    public String getConnectionStats() {
        int totalConnections = connectionPool.size();
        long healthyConnections = healthStatus.values().stream()
            .mapToLong(healthy -> healthy ? 1 : 0)
            .sum();
            
        return String.format("连接池状态: %d/%d 健康", healthyConnections, totalConnections);
    }
    
    /**
     * 手动刷新所有连接的健康状态
     */
    public void refreshAllConnections() {
        log.info("开始刷新所有连接的健康状态...");
        
        for (Map.Entry<String, InfluxDBService> entry : connectionPool.entrySet()) {
            String connectionKey = entry.getKey();
            InfluxDBService service = entry.getValue();
            
            boolean healthy = isConnectionHealthy(connectionKey, service);
            log.debug("连接 {} 健康状态: {}", connectionKey, healthy);
        }
        
        log.info("连接健康状态刷新完成: {}", getConnectionStats());
    }
    
    /**
     * 应用关闭时清理所有连接
     */
    @PreDestroy
    public void cleanup() {
        log.info("开始清理InfluxDB连接池...");
        
        for (Map.Entry<String, InfluxDBService> entry : connectionPool.entrySet()) {
            try {
                entry.getValue().close();
                log.debug("已关闭连接: {}", entry.getKey());
            } catch (Exception e) {
                log.warn("关闭连接时出错 {}: {}", entry.getKey(), e.getMessage());
            }
        }
        
        connectionPool.clear();
        healthStatus.clear();
        log.info("InfluxDB连接池清理完成");
    }
}