package com.zork.disorder.component;


import com.alibaba.fastjson.JSONObject;
import com.zork.common.dto.NormalFieldsDTO;
import com.zork.common.service.InfluxDBService;
import com.zork.disorder.model.influx.Atoota;
import lombok.extern.slf4j.Slf4j;
import org.influxdb.dto.QueryResult;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import java.text.SimpleDateFormat;
import java.util.*;
import java.util.Objects;
import java.util.stream.Collectors;

import static com.zork.common.constant.InfluxDBConstant.AUTOGEN;
import static com.zork.common.service.InfluxDBService.getResult;
import static com.zork.common.utils.DateUtil.yearMonthDayBar;
import static com.zork.common.utils.FileUtil.appendMethodB;

/**
 * @Author: Prock.Liy
 * @Date： 2021/6/1
 * @Description： 查询influxdb一分钟内数据
 */
@Slf4j
@Component
public class InfluxDisorderComponent {

    @Value("${table-keywords}")
    private String tableKeywords;

    @Value("${count-file}")
    private String countFile;

    @Value("${missing-s:15}")
    private Integer missingS;
    
    // 注入优化组件
    @Autowired
    private MultiDbConnectionManager connectionManager;
    
    @Autowired
    private MultiDbTimestampTracker timestampTracker;
    
    @Autowired
    private SmartDeduplicator deduplicator;
    
    @Autowired
    private BatchKafkaSender batchSender;


    private final SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");

    /**
     * 每日凌晨清除当天统计的count条数
     */
    @Scheduled(cron = "${clearCountCron}")
    public void clearCountData() {
        try {
            appendMethodB(countFile, "0", false);
        } catch (Exception e) {
            log.error("task Exception:{}", e.getMessage());
        }
    }

    /**
     * 优化后的查询influxdb指标数据发送至Kafka（多库并行+去重+批量发送）
     */
    @Scheduled(cron = "${schedules}")
    public void optimizedExecute() {
        try {
            log.info("开始执行InfluxDB数据采集任务...");
            
            List<String> urlList = connectionManager.getAvailableUrls();
            List<String> databaseList = connectionManager.getAvailableDatabases();
            
            // 并行处理每个URL和数据库的组合
            urlList.parallelStream().forEach(url -> {
                databaseList.parallelStream().forEach(database -> {
                    try {
                        processUrlDatabase(url.trim(), database.trim());
                    } catch (Exception e) {
                        log.error("处理URL:{}, Database:{} 失败: {}", url, database, e.getMessage());
                    }
                });
            });
            
            log.info("InfluxDB数据采集任务完成");
            
        } catch (Exception e) {
            log.error("数据采集任务异常: {}", e.getMessage(), e);
        }
    }
    
    /**
     * 处理单个URL和数据库的数据采集
     */
    private void processUrlDatabase(String url, String database) {
        try {
            // 获取复用连接
            InfluxDBService influxDBService = connectionManager.getConnection(url, database);
            
            // 获取需要查询的表名列表
            List<String> tableNameList = getTableNameList(influxDBService, database);
            
            if (tableNameList.isEmpty()) {
                log.debug("URL:{}, Database:{} 没有找到符合条件的表", url, database);
                return;
            }
            
            log.info("URL:{}, Database:{} 找到 {} 张表需要处理", url, database, tableNameList.size());
            
            // 并行处理每张表
            List<NormalFieldsDTO> allData = tableNameList.parallelStream()
                .map(tableName -> processTable(influxDBService, url, database, tableName))
                .filter(Objects::nonNull)
                .flatMap(List::stream)
                .collect(Collectors.toList());
            
            // 批量发送到Kafka
            if (!allData.isEmpty()) {
                batchSender.addData(allData);
                log.info("URL:{}, Database:{} 处理完成，收集到 {} 条数据", url, database, allData.size());
            }
            
        } catch (Exception e) {
            log.error("处理URL:{}, Database:{} 异常: {}", url, database, e.getMessage(), e);
        }
    }
    
    /**
     * 处理单张表的数据
     */
    private List<NormalFieldsDTO> processTable(InfluxDBService influxDBService, String url, String database, String tableName) {
        try {
            // 使用增量查询SQL
            String sql = timestampTracker.buildDelayTolerantQuery(url, database, tableName);
            if (sql == null) {
                log.debug("表 {} 的查询时间窗口无效，跳过", tableName);
                return Collections.emptyList();
            }
            
            log.debug("查询表 {} SQL: {}", tableName, sql);
            QueryResult.Series series = getResult(influxDBService.query(sql));
            
            if (series == null) {
                log.debug("表 {} 查询结果为空", tableName);
                return Collections.emptyList();
            }
            
            // 处理查询结果
            return processTableData(series, url, database, tableName);
            
        } catch (Exception e) {
            log.error("处理表 {} 异常: {}", tableName, e.getMessage());
            return Collections.emptyList();
        }
    }
    
    /**
     * 处理表数据，应用去重和转换
     */
    private List<NormalFieldsDTO> processTableData(QueryResult.Series series, String url, String database, String tableName) {
        // 构建列名映射
        List<String> columns = series.getColumns();
        Map<String, Integer> columnIndexMap = buildColumnIndexMap(columns);
        
        List<NormalFieldsDTO> resultList = new ArrayList<>();
        
        for (List<Object> value : series.getValues()) {
            try {
                // 构建Atoota对象
                Atoota atoota = buildAtootaFromRow(value, columnIndexMap, tableName, url);
                if (atoota == null || atoota.getId() == null) {
                    continue;
                }
                
                // 应用智能去重
                if (deduplicator.isDuplicate(url, database, tableName, atoota.getId())) {
                    continue; // 跳过重复数据
                }
                
                // 转换为DTO
                NormalFieldsDTO dto = convertToDTO(atoota);
                resultList.add(dto);
                
            } catch (Exception e) {
                log.warn("处理表 {} 的某行数据失败: {}", tableName, e.getMessage());
            }
        }
        
        log.debug("表 {} 处理完成，有效数据: {}", tableName, resultList.size());
        return resultList;
    }

    /**
     * 获取过去几分钟时间,因为数据缺失，要求减去3秒
     *
     * @param minutes 分钟
     * @return
     */
    public Date lastFewMinutesMissingS(int minutes) {
        Calendar beforeTime = Calendar.getInstance();
        beforeTime.add(Calendar.MINUTE, -minutes);
        beforeTime.add(Calendar.SECOND, -missingS);
        return beforeTime.getTime();
    }

    /**
     * 获取过去几分钟时间,因为数据缺失，要求减去3秒
     *
     * @param intervalSecond 秒数
     * @return
     */
    public Date lastFewSecondMissingS(int intervalSecond) {
        Calendar beforeTime = Calendar.getInstance();
        beforeTime.add(Calendar.SECOND, -intervalSecond);
        beforeTime.add(Calendar.SECOND, -missingS);
        return beforeTime.getTime();
    }

    /**
     * 获取当前时间减去指定秒数的时间
     *
     * @param seconds 秒数
     * @return
     */
    public Date lastFewSecond(int seconds) {
        Calendar beforeTime = Calendar.getInstance();
        beforeTime.add(Calendar.SECOND, -seconds);
        return beforeTime.getTime();
    }

    /**
     * 手动刷新所有缓存（用于重新开始同步）
     */
    public void resetAllCaches() {
        deduplicator.clearAll();
        log.info("已重置所有缓存");
    }
    
    /**
     * 获取系统状态信息
     */
    public String getSystemStats() {
        return String.format("系统状态 - %s, %s, %s", 
            connectionManager.getConnectionStats(),
            timestampTracker.getStats(),
            deduplicator.getCacheStats());
    }

    /**
     * 构建列名索引映射
     */
    private Map<String, Integer> buildColumnIndexMap(List<String> columns) {
        Map<String, Integer> columnIndexMap = new HashMap<>();
        for (int i = 0; i < columns.size(); i++) {
            columnIndexMap.put(columns.get(i), i);
        }
        return columnIndexMap;
    }
    
    /**
     * 从行数据构建Atoota对象
     */
    private Atoota buildAtootaFromRow(List<Object> value, Map<String, Integer> columnIndexMap, String tableName, String url) {
        try {
            return Atoota.builder()
                .time(getValueByColumn(value, columnIndexMap, "time"))
                .adaptive(getValueByColumn(value, columnIndexMap, "adaptive"))
                .checkpoints(getValueByColumn(value, columnIndexMap, "checkpoints"))
                .hosts(getValueByColumn(value, columnIndexMap, "hosts"))
                .id(getIdValueByColumn(value, columnIndexMap, "id"))
                .latency(getValueByColumn(value, columnIndexMap, "latency"))
                .metrics(getValueByColumn(value, columnIndexMap, "metrics"))
                .timestamps(getValueByColumn(value, columnIndexMap, "timestamps"))
                .table(tableName)
                .ip(url)
                .build();
        } catch (Exception e) {
            log.warn("构建Atoota对象失败: {}", e.getMessage());
            return null;
        }
    }
    
    @Value("${spring.kafka.producer.topic:influx-disorder-topic}")
    private String topic;

    /**
     * 转换Atoota为DTO
     */
    private NormalFieldsDTO convertToDTO(Atoota atoota) {
        return NormalFieldsDTO.builder()
            .normalFields(JSONObject.parseObject(JSONObject.toJSONString(atoota), Map.class))
            .timestamp(System.currentTimeMillis())
            .logTypeName(topic)
            .build();
    }
    
    /**
     * 根据列名安全获取值
     */
    private String getValueByColumn(List<Object> value, Map<String, Integer> columnIndexMap, String columnName) {
        Integer index = columnIndexMap.get(columnName);
        if (index == null || index >= value.size()) {
            return null;
        }
        Object val = value.get(index);
        return val == null ? null : val.toString();
    }
    
    /**
     * 根据列名安全获取ID值（数字类型）
     */
    private Long getIdValueByColumn(List<Object> value, Map<String, Integer> columnIndexMap, String columnName) {
        Integer index = columnIndexMap.get(columnName);
        if (index == null || index >= value.size()) {
            return null;
        }
        Object val = value.get(index);
        if (val == null) {
            return null;
        }
        try {
            return Math.round((double) val);
        } catch (Exception e) {
            log.warn("转换ID值失败: {}", val);
            return null;
        }
    }

    /**
     * 获取指定数据库的所有表名（支持多库）
     *
     * @param influxDBService 连接服务
     * @param database 数据库名
     * @return 表名列表
     */
    public List<String> getTableNameList(InfluxDBService influxDBService, String database) {
        List<String> tableNameList = new ArrayList<>();
        try {
            // 查询所有表名
            QueryResult result = influxDBService.query("show measurements");
            QueryResult.Series series = getResult(result);

            if (series != null && series.getValues() != null) {
                for (List<Object> value : series.getValues()) {
                    String tableName = value.get(0).toString();
                    
                    // 筛选符合条件的表
                    if (isTableMatched(tableName)) {
                        tableNameList.add(tableName);
                    }
                }
            }
            
            log.debug("数据库 {} 找到 {} 张符合条件的表", database, tableNameList.size());
            
        } catch (Exception e) {
            log.error("获取数据库 {} 的表名列表失败: {}", database, e.getMessage());
        }
        return tableNameList;
    }
    
    /**
     * 判断表名是否符合条件
     */
    private boolean isTableMatched(String tableName) {
        // 基本条件：包含关键字
        if (!tableName.contains(tableKeywords)) {
            return false;
        }
        
        // 时间条件：包含今天的日期
        String todayStr = yearMonthDayBar();
        if (!tableName.contains(todayStr)) {
            return false;
        }
        
        return true;
    }


}
