package com.zork.disorder.component;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @Author: Optimized
 * @Date: 2024
 * @Description: 多库时间戳跟踪器，为每个库的每张表独立维护查询时间戳
 */
@Slf4j
@Component
public class MultiDbTimestampTracker {
    
    @Value("${missing-s:15}")
    private Integer missingSeconds;
    
    // 存储每个库每张表的最后查询时间
    // key: host|database|table, value: timestamp
    private final Map<String, Long> lastQueryTimes = new ConcurrentHashMap<>();
    
    // 清理间隔（1小时）
    private static final long CLEANUP_INTERVAL = 3600000;
    
    /**
     * 构建增量查询SQL
     * 
     * @param url InfluxDB地址
     * @param database 数据库名
     * @param tableName 表名
     * @return 增量查询SQL
     */
    public String buildIncrementalQuery(String url, String database, String tableName) {
        String trackingKey = buildTrackingKey(url, database, tableName);
        Long lastTime = lastQueryTimes.get(trackingKey);
        
        long now = System.currentTimeMillis();
        long bufferMs = missingSeconds * 1000L; // 延迟buffer转毫秒
        
        String timeCondition;
        if (lastTime == null) {
            // 首次查询：查询最近2分钟的数据
            timeCondition = "time >= now() - 2m";
            log.debug("首次查询表 {}: {}", tableName, timeCondition);
        } else {
            // 增量查询：从上次查询时间开始，重叠15秒防止漏数据
            long startTime = lastTime - 15000; // 重叠15秒
            long endTime = now - bufferMs; // 不查最新的数据，给写入留buffer
            
            timeCondition = String.format("time >= %dms and time <= %dms", startTime, endTime);
            log.debug("增量查询表 {}: {} (从{}到{})", tableName, timeCondition, 
                new java.util.Date(startTime), new java.util.Date(endTime));
        }
        
        // 更新查询时间戳（不包含buffer，记录实际查询的结束时间）
        lastQueryTimes.put(trackingKey, now - bufferMs);
        
        return String.format("select * from \"%s\" where %s tz('Asia/Shanghai')", 
            tableName, timeCondition);
    }
    
    /**
     * 构建用于延迟容忍的查询SQL（给写入更多时间）
     */
    public String buildDelayTolerantQuery(String url, String database, String tableName) {
        String trackingKey = buildTrackingKey(url, database, tableName);
        Long lastTime = lastQueryTimes.get(trackingKey);
        
        long now = System.currentTimeMillis();
        long bufferMs = (missingSeconds + 15) * 1000L; // 额外增加15秒buffer
        
        String timeCondition;
        if (lastTime == null) {
            // 首次查询：查询最近5分钟的数据
            timeCondition = "time >= now() - 5m and time <= now() - 30s";
        } else {
            // 增量查询：重叠30秒，结束时间提前30秒
            long startTime = lastTime - 30000; // 重叠30秒
            long endTime = now - bufferMs; // 提前更多时间
            
            if (startTime >= endTime) {
                // 如果开始时间大于等于结束时间，跳过此次查询
                log.debug("查询时间窗口无效，跳过表 {} 的查询", tableName);
                return null;
            }
            
            timeCondition = String.format("time >= %dms and time <= %dms", startTime, endTime);
        }
        
        // 更新时间戳
        lastQueryTimes.put(trackingKey, now - bufferMs);
        
        return String.format("select * from \"%s\" where %s tz('Asia/Shanghai')", 
            tableName, timeCondition);
    }
    
    /**
     * 构建跟踪键：host|database|table
     */
    private String buildTrackingKey(String url, String database, String tableName) {
        String host = extractHost(url);
        return String.format("%s|%s|%s", host, database, tableName);
    }
    
    /**
     * 从URL中提取主机地址
     */
    private String extractHost(String url) {
        try {
            // http://influx1:8086 -> influx1:8086
            return url.substring(url.indexOf("://") + 3);
        } catch (Exception e) {
            return url;
        }
    }
    
    /**
     * 获取指定表的最后查询时间
     */
    public Long getLastQueryTime(String url, String database, String tableName) {
        String trackingKey = buildTrackingKey(url, database, tableName);
        return lastQueryTimes.get(trackingKey);
    }
    
    /**
     * 手动设置表的查询时间（用于重置或初始化）
     */
    public void setLastQueryTime(String url, String database, String tableName, long timestamp) {
        String trackingKey = buildTrackingKey(url, database, tableName);
        lastQueryTimes.put(trackingKey, timestamp);
        log.info("手动设置表 {} 的查询时间戳: {}", tableName, new java.util.Date(timestamp));
    }
    
    /**
     * 定时清理长时间未使用的时间戳记录
     */
    @Scheduled(fixedDelay = 3600000) // 1小时执行一次
    public void cleanupOldTimestamps() {
        long now = System.currentTimeMillis();
        int beforeSize = lastQueryTimes.size();
        
        // 清理1天前的记录
        lastQueryTimes.entrySet().removeIf(entry -> 
            (now - entry.getValue()) > 86400000); // 24小时
            
        int afterSize = lastQueryTimes.size();
        if (beforeSize - afterSize > 0) {
            log.info("清理过期时间戳记录 {} 个，剩余 {} 个", beforeSize - afterSize, afterSize);
        }
    }
    
    /**
     * 获取统计信息
     */
    public String getStats() {
        return String.format("时间戳跟踪记录数量: %d", lastQueryTimes.size());
    }
    
    /**
     * 清理指定表的时间戳（用于重新开始同步）
     */
    public void resetTable(String url, String database, String tableName) {
        String trackingKey = buildTrackingKey(url, database, tableName);
        Long oldTime = lastQueryTimes.remove(trackingKey);
        if (oldTime != null) {
            log.info("重置表 {} 的时间戳记录", tableName);
        }
    }
}