package com.zork.disorder.component;


import com.alibaba.fastjson.JSONObject;
import com.zork.common.dto.NormalFieldsDTO;
import com.zork.common.service.InfluxDBService;
import com.zork.disorder.model.influx.Atoota;
import lombok.extern.slf4j.Slf4j;
import org.influxdb.dto.QueryResult;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import java.util.*;

import static com.zork.common.service.InfluxDBService.getResult;
import static com.zork.common.utils.DateUtil.yearMonthDayBar;

/**
 * @Author: Prock.Liy
 * @Date： 2021/6/1
 * @Description： 查询influxdb一分钟内数据
 */
@Slf4j
@Component
public class InfluxDisorderComponent {

    @Value("${table-keywords}")
    private String tableKeywords;


    @Value("${time-window.query-interval-seconds:5}")
    private Integer queryIntervalSeconds;
    
    @Value("${time-window.data-delay-seconds:3}")
    private Integer dataDelaySeconds;
    
    @Value("${deduplication.enabled:true}")
    private boolean deduplicationEnabled;
    
    @Value("${deduplication.retention-days:7}")
    private int retentionDays;
    
    // 注入组件
    @Autowired
    private MultiDbConnectionManager connectionManager;
    
    @Autowired
    private BatchKafkaSender batchSender;
    
    @Autowired
    private DatabaseDeduplicationService deduplicationService;




    /**
     * 基于时间窗口的数据采集（每5秒查询过去8秒数据，用数据库去重）
     */
    @Scheduled(cron = "${schedules}")
    public void timeWindowExecute() {
        try {
            log.info("开始执行InfluxDB时间窗口数据采集任务...");
            
            List<String> urlList = connectionManager.getAvailableUrls();
            List<String> databaseList = connectionManager.getAvailableDatabases();
            
            for (String url : urlList) {
                for (String database : databaseList) {
                    try {
                        processUrlDatabaseWithTimeWindow(url.trim(), database.trim());
                    } catch (Exception e) {
                        log.error("处理URL:{}, Database:{} 失败: {}", url, database, e.getMessage());
                    }
                }
            }
            
            log.info("InfluxDB时间窗口数据采集任务完成");
            
        } catch (Exception e) {
            log.error("数据采集任务异常: {}", e.getMessage(), e);
        }
    }
    
    /**
     * 基于时间窗口的数据库处理（新方法）
     */
    private void processUrlDatabaseWithTimeWindow(String url, String database) {
        try {
            // 获取复用连接
            InfluxDBService influxDBService = connectionManager.getConnection(url, database);
            
            // 获取需要查询的表名列表
            List<String> tableNameList = getTableNameList(influxDBService, database);
            
            if (tableNameList.isEmpty()) {
                log.debug("URL:{}, Database:{} 没有找到符合条件的表", url, database);
                return;
            }
            
            log.info("URL:{}, Database:{} 找到 {} 张表需要处理", url, database, tableNameList.size());
            
            List<NormalFieldsDTO> allData = new ArrayList<>();
            for (String tableName : tableNameList) {
                List<NormalFieldsDTO> tableData = processTableWithTimeWindow(influxDBService, url, database, tableName);
                if (tableData != null && !tableData.isEmpty()) {
                    allData.addAll(tableData);
                }
            }
            
            // 批量发送到Kafka
            if (!allData.isEmpty()) {
                batchSender.addData(allData);
                log.info("URL:{}, Database:{} 处理完成，收集到 {} 条数据", url, database, allData.size());
            }
            
        } catch (Exception e) {
            log.error("处理URL:{}, Database:{} 异常: {}", url, database, e.getMessage(), e);
        }
    }
    
    /**
     * 基于时间窗口处理单张表的数据
     */
    private List<NormalFieldsDTO> processTableWithTimeWindow(InfluxDBService influxDBService, String url, String database, String tableName) {
        try {
            // 构建时间窗口查询SQL：查询过去 (queryIntervalSeconds + dataDelaySeconds) 秒到当前时间的数据
            // 这会产生重叠数据，需要通过去重来处理
            int totalSeconds = queryIntervalSeconds + dataDelaySeconds; // 8秒
            String sql = String.format("SELECT * FROM %s WHERE time >= now() - %ds AND time <= now()", 
                tableName, totalSeconds);
            
            log.debug("查询表 {} SQL: {}", tableName, sql);
            QueryResult.Series series = getResult(influxDBService.query(sql));
            
            if (series == null) {
                log.debug("表 {} 查询结果为空", tableName);
                return Collections.emptyList();
            }
            
            // 处理查询结果，包含去重逻辑
            return processTableDataWithDeduplication(series, url, database, tableName);
            
        } catch (Exception e) {
            log.error("处理表 {} 异常: {}", tableName, e.getMessage());
            return Collections.emptyList();
        }
    }
    
    /**
     * 处理表数据，包含批量数据库去重逻辑
     */
    private List<NormalFieldsDTO> processTableDataWithDeduplication(QueryResult.Series series, String url, String database, String tableName) {
        // 构建列名映射
        List<String> columns = series.getColumns();
        Map<String, Integer> columnIndexMap = buildColumnIndexMap(columns);
        
        List<Atoota> allAtootaList = new ArrayList<>();
        List<DatabaseDeduplicationService.RecordKey> recordKeys = new ArrayList<>();
        int totalCount = 0;
        
        // 第一步：构建所有数据对象和去重键
        for (List<Object> value : series.getValues()) {
            try {
                totalCount++;
                
                // 构建Atoota对象
                Atoota atoota = buildAtootaFromRow(value, columnIndexMap, tableName, url);
                if (atoota == null || atoota.getId() == null) {
                    continue;
                }
                
                allAtootaList.add(atoota);
                
                // 如果启用去重，构建去重键
                if (deduplicationEnabled) {
                    long timeMillis = extractTimeMillis(atoota.getTime());
                    DatabaseDeduplicationService.RecordKey recordKey = 
                        new DatabaseDeduplicationService.RecordKey(atoota.getId(), timeMillis, tableName, url, database);
                    recordKeys.add(recordKey);
                }
                
            } catch (Exception e) {
                log.warn("处理表 {} 的某行数据失败: {}", tableName, e.getMessage());
            }
        }
        
        // 第二步：批量去重检查（如果启用）
        Set<String> processedKeys = new HashSet<>();
        if (deduplicationEnabled && !recordKeys.isEmpty()) {
            processedKeys = deduplicationService.getProcessedRecords(recordKeys);
        }
        
        // 第三步：过滤新数据并批量插入去重记录
        List<NormalFieldsDTO> resultList = new ArrayList<>();
        List<DatabaseDeduplicationService.RecordKey> newRecordKeys = new ArrayList<>();
        int processedCount = 0;
        
        for (int i = 0; i < allAtootaList.size(); i++) {
            Atoota atoota = allAtootaList.get(i);
            
            // 检查是否是重复数据
            if (deduplicationEnabled && i < recordKeys.size()) {
                DatabaseDeduplicationService.RecordKey recordKey = recordKeys.get(i);
                String keyString = recordKey.toKey();
                
                if (processedKeys.contains(keyString)) {
                    log.trace("跳过重复记录: {}", keyString);
                    continue; // 跳过重复数据
                }
                
                newRecordKeys.add(recordKey);
            }
            
            // 转换为DTO并添加到结果
            NormalFieldsDTO dto = convertToDTO(atoota);
            resultList.add(dto);
            processedCount++;
        }
        
        // 第四步：批量插入新的去重记录
        if (deduplicationEnabled && !newRecordKeys.isEmpty()) {
            int insertedCount = deduplicationService.batchInsertRecords(newRecordKeys);
            log.debug("批量插入去重记录: 尝试={}, 成功={}", newRecordKeys.size(), insertedCount);
        }
        
        log.debug("表 {} 处理完成，总数据: {}, 处理数据: {} 条", tableName, totalCount, processedCount);
        return resultList;
    }

    /**
     * 从InfluxDB时间字符串提取毫秒级时间戳
     */
    private long extractTimeMillis(String timeStr) {
        try {
            if (timeStr == null) return 0;
            
            // InfluxDB时间格式: "2023-12-01T10:30:45.123456789Z"
            // 直接解析完整时间戳，保留毫秒精度
            return java.time.Instant.parse(timeStr).toEpochMilli();
        } catch (Exception e) {
            log.warn("解析时间戳失败: {}", timeStr);
            return System.currentTimeMillis(); // 回退到当前时间
        }
    }

    /**
     * 定期清理过期的去重记录（每天凌晨2点执行）
     */
    @Scheduled(cron = "0 0 ${deduplication.cleanup-hour:2} * * ?")
    public void cleanupExpiredRecords() {
        if (!deduplicationEnabled) {
            return;
        }
        
        try {
            int deletedCount = deduplicationService.cleanupOldRecords(retentionDays);
            log.info("清理过期去重记录完成，删除 {} 条记录", deletedCount);
        } catch (Exception e) {
            log.error("清理过期去重记录失败: {}", e.getMessage());
        }
    }

    /**
     * 获取系统状态信息
     */
    public String getSystemStats() {
        return String.format("系统状态 - %s, %s", 
            connectionManager.getConnectionStats(),
            deduplicationService.getStats());
    }

    /**
     * 构建列名索引映射
     */
    private Map<String, Integer> buildColumnIndexMap(List<String> columns) {
        Map<String, Integer> columnIndexMap = new HashMap<>();
        for (int i = 0; i < columns.size(); i++) {
            columnIndexMap.put(columns.get(i), i);
        }
        return columnIndexMap;
    }
    
    /**
     * 从行数据构建Atoota对象
     */
    private Atoota buildAtootaFromRow(List<Object> value, Map<String, Integer> columnIndexMap, String tableName, String url) {
        try {
            return Atoota.builder()
                .time(getValueByColumn(value, columnIndexMap, "time"))
                .adaptive(getValueByColumn(value, columnIndexMap, "adaptive"))
                .checkpoints(getValueByColumn(value, columnIndexMap, "checkpoints"))
                .hosts(getValueByColumn(value, columnIndexMap, "hosts"))
                .id(getIdValueByColumn(value, columnIndexMap, "id"))
                .latency(getValueByColumn(value, columnIndexMap, "latency"))
                .metrics(getValueByColumn(value, columnIndexMap, "metrics"))
                .timestamps(getValueByColumn(value, columnIndexMap, "timestamps"))
                .table(tableName)
                .ip(url)
                .build();
        } catch (Exception e) {
            log.warn("构建Atoota对象失败: {}", e.getMessage());
            return null;
        }
    }
    
    @Value("${spring.kafka.producer.topic:influx-disorder-topic}")
    private String topic;

    /**
     * 转换Atoota为DTO
     */
    @SuppressWarnings("unchecked")
    private NormalFieldsDTO convertToDTO(Atoota atoota) {
        return NormalFieldsDTO.builder()
            .normalFields((Map<String, Object>) JSONObject.parseObject(JSONObject.toJSONString(atoota), Map.class))
            .timestamp(System.currentTimeMillis())
            .logTypeName(topic)
            .build();
    }
    
    /**
     * 根据列名安全获取值
     */
    private String getValueByColumn(List<Object> value, Map<String, Integer> columnIndexMap, String columnName) {
        Integer index = columnIndexMap.get(columnName);
        if (index == null || index >= value.size()) {
            return null;
        }
        Object val = value.get(index);
        return val == null ? null : val.toString();
    }
    
    /**
     * 根据列名安全获取ID值（数字类型）
     */
    private Long getIdValueByColumn(List<Object> value, Map<String, Integer> columnIndexMap, String columnName) {
        Integer index = columnIndexMap.get(columnName);
        if (index == null || index >= value.size()) {
            return null;
        }
        Object val = value.get(index);
        if (val == null) {
            return null;
        }
        try {
            return Math.round((double) val);
        } catch (Exception e) {
            log.warn("转换ID值失败: {}", val);
            return null;
        }
    }

    /**
     * 获取指定数据库的所有表名（支持多库）
     *
     * @param influxDBService 连接服务
     * @param database 数据库名
     * @return 表名列表
     */
    public List<String> getTableNameList(InfluxDBService influxDBService, String database) {
        List<String> tableNameList = new ArrayList<>();
        try {
            // 查询所有表名
            QueryResult result = influxDBService.query("show measurements");
            QueryResult.Series series = getResult(result);

            if (series != null && series.getValues() != null) {
                for (List<Object> value : series.getValues()) {
                    String tableName = value.get(0).toString();
                    
                    // 筛选符合条件的表
                    if (isTableMatched(tableName)) {
                        tableNameList.add(tableName);
                    }
                }
            }
            
            log.debug("数据库 {} 找到 {} 张符合条件的表", database, tableNameList.size());
            
        } catch (Exception e) {
            log.error("获取数据库 {} 的表名列表失败: {}", database, e.getMessage());
        }
        return tableNameList;
    }
    
    /**
     * 判断表名是否符合条件
     */
    private boolean isTableMatched(String tableName) {
        // 基本条件：包含关键字
        if (!tableName.contains(tableKeywords)) {
            return false;
        }
        
        // 时间条件：包含今天的日期
        String todayStr = yearMonthDayBar();
        if (!tableName.contains(todayStr)) {
            return false;
        }
        
        return true;
    }


}
