package com.zorkdata.datamask.util;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.yaml.snakeyaml.Yaml;
import org.yaml.snakeyaml.constructor.SafeConstructor;

import java.io.*;
import java.net.URL;
import java.util.*;

/**
 * ClassName: LoadConf
 * Email: zhuzhigang@zork.com.cn
 * Date: 2019\6\27 0027
 *
 * @author: zhuzhigang
 **/

public class LoadConf {
    private static final Logger LOG = LoggerFactory.getLogger(com.zorkdata.datamask.util.LoadConf.class);
    public static final int DEFAULT_MAP_CAPACITY = 16;

    public LoadConf() {
    }

    public static List<URL> findResources(String name) {
        try {
            Enumeration<URL> resources = Thread.currentThread().getContextClassLoader().getResources(name);
            ArrayList ret = new ArrayList();

            while (resources.hasMoreElements()) {
                ret.add(resources.nextElement());
            }

            return ret;
        } catch (IOException var3) {
            throw new RuntimeException(var3);
        }
    }

    public static Map findAndReadYaml(String name, boolean mustExist, boolean canMultiple) {
        InputStream in = null;
        boolean confFileEmpty = false;

        try {
            in = getConfigFileInputStream(name, canMultiple);
            if (null != in) {
                Yaml yaml = new Yaml(new SafeConstructor());
                Map ret = (Map) yaml.load(new InputStreamReader(in));
                if (null != ret) {
                    HashMap var7 = new HashMap(ret);
                    return var7;
                }

                confFileEmpty = true;
            }

            if (mustExist) {
                if (confFileEmpty) {
                    throw new RuntimeException("Config file " + name + " doesn't have any valid storm configs");
                } else {
                    throw new RuntimeException("Could not find config file on classpath " + name);
                }
            } else {
                HashMap var19 = new HashMap(DEFAULT_MAP_CAPACITY);
                return var19;
            }
        } catch (IOException var17) {
            throw new RuntimeException(var17);
        } finally {
            if (null != in) {
                try {
                    in.close();
                } catch (IOException var16) {
                    throw new RuntimeException(var16);
                }
            }
        }
    }

    public static InputStream getConfigFileInputStream(String configFilePath, boolean canMultiple) throws IOException {
        if (null == configFilePath) {
            throw new IOException("Could not find config file, name not specified");
        } else {
            HashSet<URL> resources = new HashSet(findResources(configFilePath));
            if (resources.isEmpty()) {
                File configFile = new File(configFilePath);
                return configFile.exists() ? new FileInputStream(configFile) : null;
            } else if (resources.size() > 1 && !canMultiple) {
                throw new IOException("Found multiple " + configFilePath + " resources. You're probably bundling the Storm jars with your topology jar. " + resources);
            } else {
                LOG.info("Using " + configFilePath + " from resources");
                URL resource = (URL) resources.iterator().next();
                return resource.openStream();
            }
        }
    }

    public static InputStream getConfigFileInputStream(String configFilePath) throws IOException {
        return getConfigFileInputStream(configFilePath, true);
    }

    public static Map loadYaml(String confPath) {
        return findAndReadYaml(confPath, true, true);
    }

    public static Map loadProperty(String prop) {
        InputStream in = null;
        Properties properties = new Properties();

        try {
            in = getConfigFileInputStream(prop);
            properties.load(in);
        } catch (FileNotFoundException var12) {
            throw new RuntimeException("No such file " + prop);
        } catch (Exception var13) {
            throw new RuntimeException("Failed to read config file");
        } finally {
            if (null != in) {
                try {
                    in.close();
                } catch (IOException var11) {
                    throw new RuntimeException(var11);
                }
            }
        }

        Map ret = new HashMap(DEFAULT_MAP_CAPACITY);
        ret.putAll(properties);
        return ret;
    }


}
