"use client"

import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Badge } from "@/components/ui/badge"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { Checkbox } from "@/components/ui/checkbox"
import { Alert, AlertDescription } from "@/components/ui/alert"
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog"
import { RefreshCw, AlertTriangle, Target, Activity, Eye } from "lucide-react"
import { useState } from "react"

interface NetworkZone {
  id: string
  name: string
  status: string
  vmAgentCount: number
  exporterCount: number
  loadPercentage: number
  vmAgents: Array<{
    id: string
    load: number
    percentage: number
    status: string
  }>
}

interface RebalancePanelProps {
  networkZones: NetworkZone[]
}

interface OperationDetail {
  time: string
  action: string
  zone: string
  status: string
  details: string
  changes: {
    before: Array<{ vmAgent: string; exporters: number; load: number }>
    after: Array<{ vmAgent: string; exporters: number; load: number }>
    migrated: Array<{ exporter: string; from: string; to: string; type: string }>
  }
}

export default function RebalancePanel({ networkZones }: RebalancePanelProps) {
  const [selectedZone, setSelectedZone] = useState<string>("")
  const [selectedVmAgent, setSelectedVmAgent] = useState<string>("")
  const [targetVmAgent, setTargetVmAgent] = useState<string>("")
  const [selectedExporters, setSelectedExporters] = useState<string[]>([])
  const [migrationMode, setMigrationMode] = useState<"partial" | "all">("partial")
  const [isRebalancing, setIsRebalancing] = useState(false)
  const [showOperationDetail, setShowOperationDetail] = useState(false)
  const [selectedOperation, setSelectedOperation] = useState<OperationDetail | null>(null)

  const handleRebalance = async () => {
    setIsRebalancing(true)
    // Simulate rebalancing process
    setTimeout(() => {
      setIsRebalancing(false)
    }, 3000)
  }

  const getImbalancedZones = () => {
    return networkZones.filter((zone) => {
      const loads = zone.vmAgents.map((agent) => agent.percentage)
      const maxLoad = Math.max(...loads)
      const minLoad = Math.min(...loads)
      return maxLoad - minLoad > 20 // 负载差异超过20%认为不平衡
    })
  }

  const imbalancedZones = getImbalancedZones()

  // Mock exporter data for selected VmAgent
  const getExportersForVmAgent = (vmAgentId: string) => {
    const mockExporters = [
      { id: "mysql-01", type: "MySQL", host: "db-01.example.com", port: 3306 },
      { id: "redis-01", type: "Redis", host: "cache-01.example.com", port: 6379 },
      { id: "postgres-01", type: "PostgreSQL", host: "pg-01.example.com", port: 5432 },
      { id: "mongodb-01", type: "MongoDB", host: "mongo-01.example.com", port: 27017 },
      { id: "kafka-01", type: "Kafka", host: "kafka-01.example.com", port: 9092 },
    ]
    return mockExporters
  }

  const operationHistory: OperationDetail[] = [
    {
      time: "2024-01-15 14:30:25",
      action: "自动再平衡",
      zone: "华北区-01",
      status: "成功",
      details: "迁移 127 个 Exporter，负载均衡度提升 18%",
      changes: {
        before: [
          { vmAgent: "vmagent-01-1", exporters: 342, load: 85 },
          { vmAgent: "vmagent-01-2", exporters: 298, load: 72 },
          { vmAgent: "vmagent-01-3", exporters: 387, load: 91 },
        ],
        after: [
          { vmAgent: "vmagent-01-1", exporters: 309, load: 76 },
          { vmAgent: "vmagent-01-2", exporters: 331, load: 79 },
          { vmAgent: "vmagent-01-3", exporters: 387, load: 82 },
        ],
        migrated: [
          { exporter: "mysql-exporter-15", from: "vmagent-01-1", to: "vmagent-01-2", type: "MySQL" },
          { exporter: "redis-exporter-08", from: "vmagent-01-1", to: "vmagent-01-2", type: "Redis" },
          { exporter: "postgres-exporter-03", from: "vmagent-01-3", to: "vmagent-01-2", type: "PostgreSQL" },
        ],
      },
    },
    {
      time: "2024-01-15 11:15:42",
      action: "手动迁移",
      zone: "华东区-01",
      status: "成功",
      details: "从 vmagent-02-1 迁移 45 个 Exporter 到 vmagent-02-3",
      changes: {
        before: [
          { vmAgent: "vmagent-02-1", exporters: 387, load: 91 },
          { vmAgent: "vmagent-02-3", exporters: 298, load: 72 },
        ],
        after: [
          { vmAgent: "vmagent-02-1", exporters: 342, load: 85 },
          { vmAgent: "vmagent-02-3", exporters: 343, load: 78 },
        ],
        migrated: [
          { exporter: "mongodb-exporter-12", from: "vmagent-02-1", to: "vmagent-02-3", type: "MongoDB" },
          { exporter: "kafka-exporter-05", from: "vmagent-02-1", to: "vmagent-02-3", type: "Kafka" },
        ],
      },
    },
    {
      time: "2024-01-14 16:22:18",
      action: "节点扩容",
      zone: "华南区-01",
      status: "成功",
      details: "新增 vmagent-03-3 节点，自动分配 200 个 Exporter",
      changes: {
        before: [
          { vmAgent: "vmagent-03-1", exporters: 450, load: 95 },
          { vmAgent: "vmagent-03-2", exporters: 420, load: 89 },
        ],
        after: [
          { vmAgent: "vmagent-03-1", exporters: 350, load: 78 },
          { vmAgent: "vmagent-03-2", exporters: 320, load: 72 },
          { vmAgent: "vmagent-03-3", exporters: 200, load: 45 },
        ],
        migrated: [
          { exporter: "elasticsearch-exporter-01", from: "vmagent-03-1", to: "vmagent-03-3", type: "Elasticsearch" },
          { exporter: "rabbitmq-exporter-02", from: "vmagent-03-2", to: "vmagent-03-3", type: "RabbitMQ" },
        ],
      },
    },
  ]

  const handleViewOperationDetail = (operation: OperationDetail) => {
    setSelectedOperation(operation)
    setShowOperationDetail(true)
  }

  return (
    <div className="space-y-6">
      {/* Rebalance Status Alert */}
      {imbalancedZones.length > 0 && (
        <Alert className="border-yellow-200 bg-yellow-50">
          <AlertTriangle className="h-4 w-4 text-yellow-600" />
          <AlertDescription className="text-yellow-800">
            检测到 {imbalancedZones.length} 个网络区存在负载不平衡，建议执行再平衡操作。 不平衡区域:{" "}
            {imbalancedZones.map((z) => z.name).join(", ")}
          </AlertDescription>
        </Alert>
      )}

      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {/* Auto Rebalance */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center space-x-2">
              <RefreshCw className="h-5 w-5" />
              <span>智能再平衡</span>
            </CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="space-y-3">
              <div>
                <label className="text-sm font-medium text-gray-700">选择网络区</label>
                <Select value={selectedZone} onValueChange={setSelectedZone}>
                  <SelectTrigger className="mt-1">
                    <SelectValue placeholder="选择要平衡的网络区" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">所有网络区</SelectItem>
                    {networkZones.map((zone) => (
                      <SelectItem key={zone.id} value={zone.id}>
                        {zone.name}
                        {imbalancedZones.some((z) => z.id === zone.id) && (
                          <Badge variant="destructive" className="ml-2 text-xs">
                            不平衡
                          </Badge>
                        )}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-2">
                <label className="text-sm font-medium text-gray-700">平衡策略</label>
                <div className="space-y-2">
                  <div className="flex items-center space-x-2">
                    <Checkbox id="minimize-migration" defaultChecked />
                    <label htmlFor="minimize-migration" className="text-sm">
                      最小迁移成本
                    </label>
                  </div>
                  <div className="flex items-center space-x-2">
                    <Checkbox id="component-balance" defaultChecked />
                    <label htmlFor="component-balance" className="text-sm">
                      按组件类型均衡
                    </label>
                  </div>
                  <div className="flex items-center space-x-2">
                    <Checkbox id="load-threshold" />
                    <label htmlFor="load-threshold" className="text-sm">
                      考虑负载阈值
                    </label>
                  </div>
                </div>
              </div>
            </div>

            <Button onClick={handleRebalance} disabled={!selectedZone || isRebalancing} className="w-full">
              {isRebalancing ? (
                <>
                  <RefreshCw className="h-4 w-4 mr-2 animate-spin" />
                  执行中...
                </>
              ) : (
                <>
                  <RefreshCw className="h-4 w-4 mr-2" />
                  开始再平衡
                </>
              )}
            </Button>

            {selectedZone && (
              <div className="mt-4 p-3 bg-blue-50 rounded-lg">
                <h4 className="text-sm font-medium text-blue-900 mb-2">预计迁移影响</h4>
                <div className="text-xs text-blue-700 space-y-1">
                  <div>• 预计迁移 Exporter: ~150 个</div>
                  <div>• 预计停机时间: 少于30秒</div>
                  <div>• 负载均衡度提升: 15%</div>
                </div>
              </div>
            )}
          </CardContent>
        </Card>

        {/* Manual Migration */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center space-x-2">
              <Target className="h-5 w-5" />
              <span>手动迁移</span>
            </CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="space-y-3">
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="text-sm font-medium text-gray-700">源 VmAgent</label>
                  <Select value={selectedVmAgent} onValueChange={setSelectedVmAgent}>
                    <SelectTrigger className="mt-1">
                      <SelectValue placeholder="选择源节点" />
                    </SelectTrigger>
                    <SelectContent>
                      {networkZones.flatMap((zone) =>
                        zone.vmAgents.map((agent) => (
                          <SelectItem key={agent.id} value={agent.id}>
                            {agent.id} ({agent.load} Exporters, {agent.percentage}%)
                          </SelectItem>
                        )),
                      )}
                    </SelectContent>
                  </Select>
                </div>

                <div>
                  <label className="text-sm font-medium text-gray-700">目标 VmAgent</label>
                  <Select value={targetVmAgent} onValueChange={setTargetVmAgent}>
                    <SelectTrigger className="mt-1">
                      <SelectValue placeholder="选择目标节点" />
                    </SelectTrigger>
                    <SelectContent>
                      {networkZones.flatMap((zone) =>
                        zone.vmAgents
                          .filter((agent) => agent.id !== selectedVmAgent)
                          .map((agent) => (
                            <SelectItem key={agent.id} value={agent.id}>
                              {agent.id} ({agent.load} Exporters, {agent.percentage}%)
                            </SelectItem>
                          )),
                      )}
                    </SelectContent>
                  </Select>
                </div>
              </div>

              <div>
                <label className="text-sm font-medium text-gray-700">迁移范围</label>
                {selectedVmAgent ? (
                  <div className="mt-2 space-y-2 max-h-40 overflow-y-auto border rounded-md p-2">
                    {getExportersForVmAgent(selectedVmAgent).map((exporter) => (
                      <div key={exporter.id} className="flex items-center space-x-2">
                        <Checkbox
                          id={exporter.id}
                          checked={selectedExporters.includes(exporter.id)}
                          onCheckedChange={(checked) => {
                            if (checked) {
                              setSelectedExporters([...selectedExporters, exporter.id])
                            } else {
                              setSelectedExporters(selectedExporters.filter((id) => id !== exporter.id))
                            }
                          }}
                        />
                        <label htmlFor={exporter.id} className="text-sm flex-1">
                          <span className="font-medium">{exporter.id}</span>
                          <span className="text-gray-500 ml-2">({exporter.type})</span>
                          <span className="text-xs text-gray-400 block">
                            {exporter.host}:{exporter.port}
                          </span>
                        </label>
                      </div>
                    ))}
                  </div>
                ) : (
                  <div className="mt-2 p-3 bg-gray-50 rounded-md text-sm text-gray-500">请先选择源 VmAgent 节点</div>
                )}
              </div>
            </div>

            <Button
              disabled={!selectedVmAgent || !targetVmAgent || selectedExporters.length === 0}
              variant="outline"
              className="w-full bg-transparent"
            >
              <Target className="h-4 w-4 mr-2" />
              执行迁移
            </Button>

            {selectedVmAgent && targetVmAgent && selectedExporters.length > 0 && (
              <div className="mt-4 p-3 bg-green-50 rounded-lg">
                <h4 className="text-sm font-medium text-green-900 mb-2">迁移预览</h4>
                <div className="text-xs text-green-700 space-y-1">
                  <div>• 迁移数量: {selectedExporters.length} 个 Exporter</div>
                  <div>• 预计时间: {Math.ceil(selectedExporters.length / 30)} 分钟</div>
                  <div>• 影响范围: 最小化</div>
                </div>
              </div>
            )}
          </CardContent>
        </Card>
      </div>

      {/* Recent Operations */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center space-x-2">
            <Activity className="h-5 w-5" />
            <span>最近操作记录</span>
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="space-y-3">
            {operationHistory.map((operation, index) => (
              <div
                key={index}
                className="flex items-start justify-between p-3 bg-gray-50 rounded-lg hover:bg-gray-100 transition-colors"
              >
                <div className="space-y-1 flex-1">
                  <div className="flex items-center space-x-2">
                    <Badge variant={operation.status === "成功" ? "default" : "destructive"} className="text-xs">
                      {operation.status}
                    </Badge>
                    <span className="text-sm font-medium">{operation.action}</span>
                    <span className="text-xs text-gray-500">{operation.zone}</span>
                  </div>
                  <p className="text-xs text-gray-600">{operation.details}</p>
                </div>
                <div className="flex items-center space-x-2">
                  <span className="text-xs text-gray-400">{operation.time}</span>
                  <Button
                    variant="ghost"
                    size="sm"
                    onClick={() => handleViewOperationDetail(operation)}
                    className="h-6 w-6 p-0"
                  >
                    <Eye className="h-3 w-3" />
                  </Button>
                </div>
              </div>
            ))}
          </div>
        </CardContent>
      </Card>

      <Dialog open={showOperationDetail} onOpenChange={setShowOperationDetail}>
        <DialogContent className="max-w-4xl max-h-[80vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>操作详情 - {selectedOperation?.action}</DialogTitle>
            <DialogDescription>
              {selectedOperation?.zone} | {selectedOperation?.time} | {selectedOperation?.status}
            </DialogDescription>
          </DialogHeader>

          {selectedOperation && (
            <div className="space-y-6">
              {/* Operation Summary */}
              <div className="p-4 bg-gray-50 rounded-lg">
                <h4 className="font-medium mb-2">操作概要</h4>
                <p className="text-sm text-gray-600">{selectedOperation.details}</p>
              </div>

              {/* Before/After Comparison */}
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <h4 className="font-medium mb-3 text-red-600">同步前配置</h4>
                  <div className="space-y-2">
                    {selectedOperation.changes.before.map((config, index) => (
                      <div key={index} className="p-3 bg-red-50 rounded border border-red-200">
                        <div className="font-medium text-sm">{config.vmAgent}</div>
                        <div className="text-xs text-gray-600">
                          Exporters: {config.exporters} | 负载: {config.load}%
                        </div>
                      </div>
                    ))}
                  </div>
                </div>

                <div>
                  <h4 className="font-medium mb-3 text-green-600">同步后配置</h4>
                  <div className="space-y-2">
                    {selectedOperation.changes.after.map((config, index) => (
                      <div key={index} className="p-3 bg-green-50 rounded border border-green-200">
                        <div className="font-medium text-sm">{config.vmAgent}</div>
                        <div className="text-xs text-gray-600">
                          Exporters: {config.exporters} | 负载: {config.load}%
                        </div>
                      </div>
                    ))}
                  </div>
                </div>
              </div>

              {/* Migration Details */}
              <div>
                <h4 className="font-medium mb-3">迁移详情</h4>
                <div className="space-y-2">
                  {selectedOperation.changes.migrated.map((migration, index) => (
                    <div
                      key={index}
                      className="flex items-center justify-between p-3 bg-blue-50 rounded border border-blue-200"
                    >
                      <div className="flex items-center space-x-3">
                        <Badge variant="outline" className="text-xs">
                          {migration.type}
                        </Badge>
                        <span className="text-sm font-medium">{migration.exporter}</span>
                      </div>
                      <div className="text-sm text-gray-600">
                        {migration.from} → {migration.to}
                      </div>
                    </div>
                  ))}
                </div>
              </div>

              {/* Impact Analysis */}
              <div className="grid grid-cols-3 gap-4">
                <div className="p-3 bg-blue-50 rounded-lg">
                  <div className="text-sm text-gray-600">迁移数量</div>
                  <div className="text-lg font-semibold text-blue-600">{selectedOperation.changes.migrated.length}</div>
                  <div className="text-xs text-gray-500">个 Exporter</div>
                </div>
                <div className="p-3 bg-green-50 rounded-lg">
                  <div className="text-sm text-gray-600">负载改善</div>
                  <div className="text-lg font-semibold text-green-600">
                    {selectedOperation.action.includes("再平衡") ? "18%" : "12%"}
                  </div>
                  <div className="text-xs text-gray-500">均衡度提升</div>
                </div>
                <div className="p-3 bg-purple-50 rounded-lg">
                  <div className="text-sm text-gray-600">执行时间</div>
                  <div className="text-lg font-semibold text-purple-600">
                    {selectedOperation.action.includes("扩容") ? "5分钟" : "2分钟"}
                  </div>
                  <div className="text-xs text-gray-500">总耗时</div>
                </div>
              </div>
            </div>
          )}

          <DialogFooter>
            <Button variant="outline" onClick={() => setShowOperationDetail(false)}>
              关闭
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  )
}
