import { notification } from 'antd';
import Cookies from 'js-cookie';

// 不同环境下获取token的方式不通
let accessToken: string = Cookies.get('OO-access_token')
  ? 'Bearer ' + JSON.parse(Cookies.get('OO-access_token')).content
  : '';

if (process.env.NODE_ENV === 'development') {
  accessToken = 'bearer e30faf14-190c-4db9-b553-b7f2097da875';
}

export class ResponseError extends Error {
  public response: Response;
  constructor(response: Response) {
    super(response.statusText);
    this.response = response;
  }
}

function checkStatus(response: any) {
  if (response && response.status >= 200 && response.status < 400) {
    if (
      response &&
      response.result &&
      response.result.status &&
      response.result.status !== 0
    ) {
      notification.error({
        message: `请求错误 ${response.result.status}`,
        description: response.result.message,
      });
    }
    return response;
  }
  if (response && response.status === 401) {
    Cookies.remove('OO-access_token');
    Cookies.remove('OO-refresh_userInfo');
    let domains = document.domain.split('.');
    if (domains.length > 0) {
      domains[0] = 'paas';
    }
    let domain = domains.join('.');

    const href =
      document.location.protocol +
      '//' +
      domain +
      '#/login/?r_url=' +
      window.location.href;

    window.location.replace(href);
  }
  if (!response) {
    notification.error({
      description: '您的网络发生异常，无法连接服务器',
      message: '网络异常',
    });
  } else {
    const { status, url } = response;
    if (status === 400) {
      notification.error({
        message: `请求错误，请检查参数`,
        description: '',
      });
    } else {
      notification.error({
        message: `请求错误 ${status}: ${url}`,
        description: '',
      });
    }
    const error = new ResponseError(response);
    error.response = response;
    throw error;
  }
  return response;
}

function parseJSON(response: Response) {
  if (response.status === 204 || response.status === 205) {
    return null;
  }
  return response.json();
}

export async function request(
  url: string,
  options?: RequestInit,
): Promise<{} | { err: ResponseError }> {
  const fetchResponse = await fetch(url, options);
  const response = checkStatus(fetchResponse);
  return parseJSON(response);
}

export function post(
  url: string,
  params: Record<string, any>,
  options?: Record<string, any>,
): Promise<{} | { err: ResponseError }> {
  return request(url, {
    method: 'POST',
    headers: {
      Authorization: accessToken,
      Accept: 'application/json',
      'Content-Type': 'application/json',
      ...options,
    },
    body: JSON.stringify(params),
  });
}

export function postForm(
  url: string,
  params: any,
  options?: Record<string, any>,
): Promise<{} | { err: ResponseError }> {
  return request(url, {
    method: 'POST',
    headers: {
      Authorization: accessToken,
      Accept: 'application/json',
      'Content-Type': 'application/x-www-form-urlencoded',
      ...options,
    },
    body: params,
  });
}

export function postFile(
  url: string,
  params: any,
): Promise<{} | { err: ResponseError }> {
  return request(url, {
    method: 'POST',
    headers: {
      Authorization: accessToken,
      Accept: 'application/json',
    },
    body: params,
  });
}

export function get(
  url: string,
  params?: Record<string, any>,
  options?: Record<string, any>,
): Promise<{} | { err: ResponseError }> {
  if (params) {
    const paramsArray: string[] = [];
    //拼接参数
    Object.keys(params).forEach(key =>
      paramsArray.push(key + '=' + params[key]),
    );
    if (url.search(/\?/) === -1) {
      url += '?' + paramsArray.join('&');
    } else {
      url += '&' + paramsArray.join('&');
    }
  }
  return request(url, {
    method: 'GET',
    headers: {
      Authorization: accessToken,
      Accept: 'application/json',
      'Content-Type': 'application/json',
      ...options,
    },
  });
}

export function del(
  url: string,
  options?: Record<string, any>,
): Promise<{} | { err: ResponseError }> {
  return request(url, {
    method: 'DELETE',
    headers: {
      Authorization: accessToken,
      Accept: 'application/json',
      'Content-Type': 'application/json',
      ...options,
    },
  });
}


