"use client"

import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Badge } from "@/components/ui/badge"
import { DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuTrigger } from "@/components/ui/dropdown-menu"
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import {
  ChevronDown,
  ChevronUp,
  AlertTriangle,
  CheckCircle,
  RefreshCw,
  MoreHorizontal,
  Play,
  Square,
  ArrowRightLeft,
  Eye,
} from "lucide-react"
import { useState } from "react"

interface VmAgent {
  id: string
  load: number
  percentage: number
  status: "normal" | "high" | "warning"
  host: string
  cpu: number
  memory: number
}

interface NetworkZone {
  id: string
  name: string
  status: "normal" | "warning" | "error"
  vmAgentCount: number
  exporterCount: number
  loadPercentage: number
  vmAgents: VmAgent[]
}

interface NetworkZoneCardProps {
  zone: NetworkZone
}

export default function NetworkZoneCard({ zone }: NetworkZoneCardProps) {
  const [isExpanded, setIsExpanded] = useState(true)
  const [showMigrationDialog, setShowMigrationDialog] = useState(false)
  const [showDetailsDialog, setShowDetailsDialog] = useState(false)
  const [selectedAgent, setSelectedAgent] = useState<VmAgent | null>(null)
  const [targetAgent, setTargetAgent] = useState<string>("")

  const getStatusColor = (status: string) => {
    switch (status) {
      case "normal":
        return "text-green-600 bg-green-100"
      case "warning":
        return "text-yellow-600 bg-yellow-100"
      case "high":
        return "text-red-600 bg-red-100"
      default:
        return "text-gray-600 bg-gray-100"
    }
  }

  const getStatusIcon = (status: string) => {
    switch (status) {
      case "normal":
        return <CheckCircle className="h-4 w-4" />
      case "warning":
      case "high":
        return <AlertTriangle className="h-4 w-4" />
      default:
        return <CheckCircle className="h-4 w-4" />
    }
  }

  const getProgressColor = (percentage: number) => {
    if (percentage >= 80) return "bg-red-500"
    if (percentage >= 60) return "bg-yellow-500"
    return "bg-green-500"
  }

  const handleNodeOperation = (agent: VmAgent, operation: "start" | "stop" | "migrate" | "details") => {
    if (operation === "migrate") {
      setSelectedAgent(agent)
      setShowMigrationDialog(true)
    } else if (operation === "details") {
      setSelectedAgent(agent)
      setShowDetailsDialog(true)
    } else {
      console.log(`[v0] ${operation} operation for ${agent.id}`)
      // TODO: Implement start/stop operations
    }
  }

  const handleMigration = () => {
    console.log(`[v0] Migrating from ${selectedAgent?.id} to ${targetAgent}`)
    setShowMigrationDialog(false)
    setSelectedAgent(null)
    setTargetAgent("")
    // TODO: Implement migration logic
  }

  const getAvailableTargets = () => {
    const allAgents = zone.vmAgents.filter((agent) => agent.id !== selectedAgent?.id)
    return allAgents
  }

  return (
    <>
      <Card
        className={`transition-all duration-200 ${zone.status === "warning" ? "border-yellow-200 bg-yellow-50" : ""}`}
      >
        <CardHeader className="pb-3">
          <div className="flex items-center justify-between">
            <div className="flex items-center space-x-2">
              <div className={`p-1 rounded-full ${getStatusColor(zone.status)}`}>{getStatusIcon(zone.status)}</div>
              <CardTitle className="text-lg">{zone.name}</CardTitle>
            </div>
            <Button variant="ghost" size="sm" onClick={() => setIsExpanded(!isExpanded)}>
              {isExpanded ? <ChevronUp className="h-4 w-4" /> : <ChevronDown className="h-4 w-4" />}
            </Button>
          </div>
        </CardHeader>

        <CardContent className="space-y-4">
          {/* Zone Summary */}
          <div className="flex items-center justify-between text-sm">
            <span className="text-gray-600">VmAgent</span>
            <Badge variant="secondary">{zone.vmAgentCount}/3 VmAgent</Badge>
          </div>

          <div className="flex items-center justify-between text-sm">
            <span className="text-gray-600">Exporter 数量</span>
            <span className="font-medium">{zone.exporterCount} 容器</span>
          </div>

          {/* Load Progress */}
          <div className="space-y-2">
            <div className="flex items-center justify-between text-sm">
              <span className="text-gray-600">平均负载</span>
              <span className="font-medium">{zone.loadPercentage}%</span>
            </div>
            <div className="w-full bg-gray-200 rounded-full h-2">
              <div
                className={`h-2 rounded-full transition-all duration-300 ${getProgressColor(zone.loadPercentage)}`}
                style={{ width: `${zone.loadPercentage}%` }}
              />
            </div>
          </div>

          {/* Action Buttons */}
          <div className="flex space-x-2 pt-2">
            <Button variant="outline" size="sm" className="flex-1 bg-transparent">
              <RefreshCw className="h-3 w-3 mr-1" />
              再平衡
            </Button>
          </div>

          {/* Expanded Details */}
          {isExpanded && (
            <div className="mt-4 pt-4 border-t space-y-3">
              <h4 className="font-medium text-sm text-gray-700">VmAgent 节点</h4>
              <div className="max-h-64 overflow-y-auto space-y-3">
                {zone.vmAgents.map((agent) => (
                  <div key={agent.id} className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                    <div className="flex items-center space-x-3">
                      <div
                        className={`w-2 h-2 rounded-full ${
                          agent.status === "normal"
                            ? "bg-green-500"
                            : agent.status === "high"
                              ? "bg-red-500"
                              : "bg-yellow-500"
                        }`}
                      />
                      <div className="flex flex-col">
                        <span className="text-sm font-medium">{agent.id}</span>
                        <span className="text-xs text-gray-500">{agent.host}</span>
                      </div>
                    </div>
                    <div className="flex items-center space-x-2">
                      <div className="text-right">
                        <div className="flex items-center space-x-1">
                          <Badge variant="outline" className="text-xs">
                            {agent.load}
                          </Badge>
                          <span className="text-xs text-gray-500">{agent.percentage}%</span>
                        </div>
                        <div className="text-xs text-gray-400">
                          CPU: {agent.cpu}% 内存: {agent.memory}%
                        </div>
                      </div>
                      {/* Dropdown Menu for node operations */}
                      <DropdownMenu>
                        <DropdownMenuTrigger asChild>
                          <Button variant="ghost" size="sm">
                            <MoreHorizontal className="h-3 w-3" />
                          </Button>
                        </DropdownMenuTrigger>
                        <DropdownMenuContent align="end">
                          <DropdownMenuItem onClick={() => handleNodeOperation(agent, "details")}>
                            <Eye className="h-3 w-3 mr-2" />
                            查看详情
                          </DropdownMenuItem>
                          <DropdownMenuItem onClick={() => handleNodeOperation(agent, "start")}>
                            <Play className="h-3 w-3 mr-2" />
                            启动
                          </DropdownMenuItem>
                          <DropdownMenuItem onClick={() => handleNodeOperation(agent, "stop")}>
                            <Square className="h-3 w-3 mr-2" />
                            停止
                          </DropdownMenuItem>
                          <DropdownMenuItem onClick={() => handleNodeOperation(agent, "migrate")}>
                            <ArrowRightLeft className="h-3 w-3 mr-2" />
                            迁移
                          </DropdownMenuItem>
                        </DropdownMenuContent>
                      </DropdownMenu>
                    </div>
                  </div>
                ))}
              </div>

              <div className="pt-2">
                <h4 className="font-medium text-sm text-gray-700 mb-2">Exporter主机负载</h4>
                <div className="grid grid-cols-1 gap-2 text-xs">
                  {zone.vmAgents.map((agent, index) => (
                    <div key={agent.id} className="flex justify-between">
                      <span>{agent.host}</span>
                      <span>
                        {agent.load} 容器 CPU: {agent.cpu}% 内存: {agent.memory}%
                      </span>
                    </div>
                  ))}
                </div>
              </div>
            </div>
          )}
        </CardContent>
      </Card>

      {/* Migration Dialog */}
      <Dialog open={showMigrationDialog} onOpenChange={setShowMigrationDialog}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>节点迁移</DialogTitle>
            <DialogDescription>将 {selectedAgent?.id} 的 Exporter 迁移到其他 VmAgent 节点</DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div>
              <label className="text-sm font-medium">源节点</label>
              <div className="mt-1 p-2 bg-gray-100 rounded text-sm">
                {selectedAgent?.id} ({selectedAgent?.host})
              </div>
            </div>
            <div>
              <label className="text-sm font-medium">目标节点</label>
              <Select value={targetAgent} onValueChange={setTargetAgent}>
                <SelectTrigger className="mt-1">
                  <SelectValue placeholder="选择目标节点" />
                </SelectTrigger>
                <SelectContent>
                  {getAvailableTargets().map((agent) => (
                    <SelectItem key={agent.id} value={agent.id}>
                      {agent.id} ({agent.host}) - 负载: {agent.percentage}%
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => setShowMigrationDialog(false)}>
              取消
            </Button>
            <Button onClick={handleMigration} disabled={!targetAgent}>
              开始迁移
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* VmAgent Details Dialog */}
      <Dialog open={showDetailsDialog} onOpenChange={setShowDetailsDialog}>
        <DialogContent className="max-w-4xl max-h-[80vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>VmAgent 详情 - {selectedAgent?.id}</DialogTitle>
            <DialogDescription>查看 VmAgent 的配置文件和历史版本</DialogDescription>
          </DialogHeader>
          <VmAgentDetailsContent agent={selectedAgent} />
          <DialogFooter>
            <Button variant="outline" onClick={() => setShowDetailsDialog(false)}>
              关闭
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </>
  )
}

// VmAgent Details Component
function VmAgentDetailsContent({ agent }: { agent: VmAgent | null }) {
  const [selectedVersion, setSelectedVersion] = useState("latest")

  if (!agent) return null

  const configVersions = [
    { id: "latest", name: "最新版本", date: "2024-01-15 14:30:25", status: "active" },
    { id: "v1.2.3", name: "v1.2.3", date: "2024-01-14 09:15:42", status: "history" },
    { id: "v1.2.2", name: "v1.2.2", date: "2024-01-13 16:22:18", status: "history" },
    { id: "v1.2.1", name: "v1.2.1", date: "2024-01-12 11:45:33", status: "history" },
  ]

  const currentConfig = `# VmAgent Configuration - ${selectedVersion}
global:
  scrape_interval: 15s
  evaluation_interval: 15s

remote_write:
  - url: http://vmserver-cluster:8428/api/v1/write
    queue_config:
      max_samples_per_send: 10000
      batch_send_deadline: 5s

scrape_configs:
  - job_name: 'mysql-exporters'
    static_configs:
      - targets: ['mysql-exporter-1:9104', 'mysql-exporter-2:9104']
    scrape_interval: 30s
    metrics_path: /metrics

  - job_name: 'redis-exporters'
    static_configs:
      - targets: ['redis-exporter-1:9121', 'redis-exporter-2:9121']
    scrape_interval: 30s

  - job_name: 'postgresql-exporters'
    static_configs:
      - targets: ['postgres-exporter-1:9187']
    scrape_interval: 30s

rule_files:
  - "alert_rules.yml"

alerting:
  alertmanagers:
    - static_configs:
        - targets: ['alertmanager:9093']`

  return (
    <div className="space-y-6">
      {/* Agent Info */}
      <div className="grid grid-cols-2 gap-4 p-4 bg-gray-50 rounded-lg">
        <div>
          <span className="text-sm text-gray-600">节点ID:</span>
          <span className="ml-2 font-medium">{agent.id}</span>
        </div>
        <div>
          <span className="text-sm text-gray-600">主机:</span>
          <span className="ml-2 font-medium">{agent.host}</span>
        </div>
        <div>
          <span className="text-sm text-gray-600">当前负载:</span>
          <span className="ml-2 font-medium">
            {agent.load} Exporters ({agent.percentage}%)
          </span>
        </div>
        <div>
          <span className="text-sm text-gray-600">状态:</span>
          <Badge
            className={`ml-2 ${
              agent.status === "normal"
                ? "bg-green-100 text-green-800"
                : agent.status === "warning"
                  ? "bg-yellow-100 text-yellow-800"
                  : "bg-red-100 text-red-800"
            }`}
          >
            {agent.status === "normal" ? "正常" : agent.status === "warning" ? "警告" : "高负载"}
          </Badge>
        </div>
      </div>

      {/* Version Selector */}
      <div className="space-y-2">
        <label className="text-sm font-medium">配置版本</label>
        <Select value={selectedVersion} onValueChange={setSelectedVersion}>
          <SelectTrigger>
            <SelectValue />
          </SelectTrigger>
          <SelectContent>
            {configVersions.map((version) => (
              <SelectItem key={version.id} value={version.id}>
                <div className="flex items-center justify-between w-full">
                  <span>{version.name}</span>
                  <span className="text-xs text-gray-500 ml-4">{version.date}</span>
                  {version.status === "active" && (
                    <Badge variant="secondary" className="ml-2 text-xs">
                      当前
                    </Badge>
                  )}
                </div>
              </SelectItem>
            ))}
          </SelectContent>
        </Select>
      </div>

      {/* Configuration Content */}
      <div className="space-y-2">
        <label className="text-sm font-medium">配置文件内容</label>
        <div className="bg-gray-900 text-gray-100 p-4 rounded-lg overflow-x-auto">
          <pre className="text-sm whitespace-pre-wrap">{currentConfig}</pre>
        </div>
      </div>

      {/* Configuration Summary */}
      <div className="grid grid-cols-3 gap-4">
        <div className="p-3 bg-blue-50 rounded-lg">
          <div className="text-sm text-gray-600">监控目标</div>
          <div className="text-lg font-semibold text-blue-600">{agent.load}</div>
          <div className="text-xs text-gray-500">个 Exporter</div>
        </div>
        <div className="p-3 bg-green-50 rounded-lg">
          <div className="text-sm text-gray-600">采集间隔</div>
          <div className="text-lg font-semibold text-green-600">30s</div>
          <div className="text-xs text-gray-500">平均间隔</div>
        </div>
        <div className="p-3 bg-purple-50 rounded-lg">
          <div className="text-sm text-gray-600">上次更新</div>
          <div className="text-lg font-semibold text-purple-600">2小时前</div>
          <div className="text-xs text-gray-500">配置同步</div>
        </div>
      </div>
    </div>
  )
}
