"use client"

import { Card, CardContent } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "@/components/ui/dialog"
import { Activity, Server, Database, CheckCircle, HardDrive, AlertTriangle } from "lucide-react"
import { useEffect, useState } from "react"
import NetworkZoneCard from "@/components/network-zone-card"
import ExporterDistribution from "@/components/exporter-distribution"
import RebalancePanel from "@/components/rebalance-panel"
import { get, post, postForm, postFile } from '@/fetch/index';

export default function VmAgentDashboard() {
  const [showGlobalRebalanceDialog, setShowGlobalRebalanceDialog] = useState(false)

  const overviewStats = {
    networkZones: 3,
    vmAgentNodes: { active: 8, total: 9 },
    exporterCount: 2761,
    systemStatus: "正常运行",
  }

  const networkZones = [
    {
      id: "north-01",
      name: "华北区-01",
      status: "normal",
      vmAgentCount: 3,
      exporterCount: 1027,
      loadPercentage: 83,
      vmAgents: [
        {
          id: "VmAgent-01-1",
          load: 342,
          percentage: 33,
          status: "normal",
          host: "node-01.beijing",
          cpu: 45,
          memory: 72,
        },
        {
          id: "VmAgent-01-2",
          load: 298,
          percentage: 29,
          status: "normal",
          host: "node-02.beijing",
          cpu: 38,
          memory: 65,
        },
        { id: "VmAgent-01-3", load: 387, percentage: 38, status: "high", host: "node-03.beijing", cpu: 78, memory: 85 },
      ],
    },
    {
      id: "east-01",
      name: "华东区-01",
      status: "warning",
      vmAgentCount: 3,
      exporterCount: 1049,
      loadPercentage: 65,
      vmAgents: [
        {
          id: "VmAgent-02-1",
          load: 456,
          percentage: 43,
          status: "high",
          host: "node-01.shanghai",
          cpu: 82,
          memory: 78,
        },
        {
          id: "VmAgent-02-2",
          load: 293,
          percentage: 28,
          status: "normal",
          host: "node-02.shanghai",
          cpu: 42,
          memory: 58,
        },
        {
          id: "VmAgent-02-3",
          load: 300,
          percentage: 29,
          status: "normal",
          host: "node-03.shanghai",
          cpu: 45,
          memory: 62,
        },
      ],
    },
    {
      id: "south-01",
      name: "华南区-01",
      status: "normal",
      vmAgentCount: 2,
      exporterCount: 685,
      loadPercentage: 80,
      vmAgents: [
        {
          id: "VmAgent-03-1",
          load: 342,
          percentage: 50,
          status: "normal",
          host: "node-01.guangzhou",
          cpu: 55,
          memory: 68,
        },
        {
          id: "VmAgent-03-2",
          load: 343,
          percentage: 50,
          status: "normal",
          host: "node-02.guangzhou",
          cpu: 58,
          memory: 71,
        },
      ],
    },
  ]
  const [mode, setMode] = useState<string>('');

  const getUserInfo = async () => {
    try {
      const res: any = await get('/api/onlyone/authcenter/user/info')
      const { code, data } = res;
      if (0 !== code) {
        return;
      }
      let displayName = data.sysUser.displayName
        ? decodeURIComponent(data.sysUser.displayName)
        : data.sysUser.username;
      data.sysUser.displayName = displayName.slice(0, 1).toUpperCase();
      setUserinfo(data.sysUser);
    } catch (error) {
      console.log(error);
    }
  };

  const getAppInfo = async () => {
    try {
      const res: any = await get(
        `/api/onlyone/authcenter/v1/system/getByAppCode/${appCode}`,
      );
      const { code, data } = res;
      if (0 !== code) {
        return;
      }
      data.appIcon = data.appIcon.substring(data.appIcon.indexOf('_:/') + 3);
      setAppInfo(data);
      const link = document.createElement('link');
      link.rel = 'shortcut icon';
      link.href = data.appIcon;
      document.head.append(link);

      const title = document.createElement('title');
      title.text = data.displayName;
      document.head.append(title);
    } catch (error) {
      console.log(error);
    }
  };

  const initPaasInfo = async () => {
    await getAppInfo();
    await getUserInfo();
  }

  useEffect(() => {
    initPaasInfo();
  }, []);

  const handleGlobalRebalance = () => {
    setShowGlobalRebalanceDialog(false)
    // TODO: Implement global rebalancing logic
    console.log("[v0] Starting global rebalance...")
  }

  return (
    <div className="min-h-screen bg-gray-50 p-6">
      <div className="max-w-7xl mx-auto space-y-6">
        {/* Header */}
        <div className="flex items-center justify-between">
          <div className="flex items-center space-x-3">
            <div className="p-2 bg-blue-100 rounded-lg">
              <Activity className="h-6 w-6 text-blue-600" />
            </div>
            <div>
              <h1 className="text-2xl font-bold text-gray-900">VmAgent 负载均衡管理</h1>
              <p className="text-gray-600">监控和管理各网络区 VmAgent 集群</p>
            </div>
          </div>
          <div className="flex items-center space-x-3">
            <Dialog open={showGlobalRebalanceDialog} onOpenChange={setShowGlobalRebalanceDialog}>
              <DialogTrigger asChild>
                <Button variant="outline" size="sm">
                  全局再平衡
                </Button>
              </DialogTrigger>
              <DialogContent>
                <DialogHeader>
                  <DialogTitle className="flex items-center space-x-2">
                    <AlertTriangle className="h-5 w-5 text-amber-500" />
                    <span>全局再平衡确认</span>
                  </DialogTitle>
                  <DialogDescription className="text-left">
                    全局再平衡将自动采用按组件类型均衡同时考虑负载阈值的平衡策略进行迁移，对配置影响较大，请谨慎操作！
                  </DialogDescription>
                </DialogHeader>
                <DialogFooter>
                  <Button variant="outline" onClick={() => setShowGlobalRebalanceDialog(false)}>
                    取消
                  </Button>
                  <Button variant="destructive" onClick={handleGlobalRebalance}>
                    确认执行
                  </Button>
                </DialogFooter>
              </DialogContent>
            </Dialog>
          </div>
        </div>

        {/* Overview Stats */}
        <div className="grid grid-cols-1 md:grid-cols-4 gap-6">
          <Card>
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-gray-600">网络区数量</p>
                  <p className="text-3xl font-bold text-gray-900">{overviewStats.networkZones}</p>
                  <p className="text-xs text-gray-500">3个区域运行</p>
                </div>
                <div className="p-3 bg-blue-100 rounded-full">
                  <Server className="h-6 w-6 text-blue-600" />
                </div>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-gray-600">VmAgent 节点</p>
                  <p className="text-3xl font-bold text-gray-900">
                    {overviewStats.vmAgentNodes.active}/{overviewStats.vmAgentNodes.total}
                  </p>
                  <p className="text-xs text-gray-500">运行/总数</p>
                </div>
                <div className="p-3 bg-green-100 rounded-full">
                  <Database className="h-6 w-6 text-green-600" />
                </div>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-gray-600">Exporter 总数</p>
                  <p className="text-3xl font-bold text-gray-900">{overviewStats.exporterCount.toLocaleString()}</p>
                  <p className="text-xs text-gray-500">正在采集</p>
                </div>
                <div className="p-3 bg-purple-100 rounded-full">
                  <HardDrive className="h-6 w-6 text-purple-600" />
                </div>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-gray-600">系统状态</p>
                  <div className="flex items-center space-x-2 mt-1">
                    <CheckCircle className="h-5 w-5 text-green-500" />
                    <span className="text-lg font-semibold text-green-600">正常运行</span>
                  </div>
                  <p className="text-xs text-gray-500">实时监控中</p>
                </div>
                <div className="p-3 bg-green-100 rounded-full">
                  <Activity className="h-6 w-6 text-green-600" />
                </div>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Main Content Tabs */}
        <Tabs defaultValue="overview" className="space-y-6">
          <TabsList className="grid w-full grid-cols-3">
            <TabsTrigger value="overview">网络区概览</TabsTrigger>
            <TabsTrigger value="distribution">负载分析</TabsTrigger>
            <TabsTrigger value="operations">运维操作</TabsTrigger>
          </TabsList>

          <TabsContent value="overview" className="space-y-6">
            <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
              {networkZones.map((zone) => (
                <NetworkZoneCard key={zone.id} zone={zone} />
              ))}
            </div>
          </TabsContent>

          <TabsContent value="distribution" className="space-y-6">
            <ExporterDistribution />
          </TabsContent>

          <TabsContent value="operations" className="space-y-6">
            <RebalancePanel networkZones={networkZones} />
          </TabsContent>
        </Tabs>
      </div>
    </div>
  )
}
