package com.zorkdata.desensitization.function;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.TypeReference;
import com.zorkdata.desensitization.config.JobConfig;
import com.zorkdata.desensitization.constans.GeneralConstants;
import com.zorkdata.desensitization.constans.ZorkdataFlag;
import com.zorkdata.desensitization.schmea.LogData;
import com.zorkdata.desensitization.utils.DateUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.flink.api.common.functions.RichFlatMapFunction;
import org.apache.flink.api.java.tuple.Tuple2;
import org.apache.flink.util.Collector;

import java.util.*;


/**
 * @author: LiaoMingtao
 * @date: 2021/2/24
 */
@Slf4j
public class DesensitizationFunction<T, R> extends RichFlatMapFunction<Tuple2<Object, Object>, LogData> {

    private JobConfig jobConfig;

    private boolean hasMatchHostname;

    public DesensitizationFunction(JobConfig jobConfig) {
        this.jobConfig = jobConfig;
        String matchHostname = jobConfig.getMatchHostname();
        this.hasMatchHostname = !"*".equals(matchHostname);
    }

    @Override
    public void flatMap(Tuple2<Object, Object> value, Collector<LogData> collector) throws Exception {
        if (null == value.getField(0).toString()) {
            log.error("存在空数据：{}", value.toString());
            return;
        }
        try {
            LogData logData = JSON.parseObject(value.getField(0).toString(), new TypeReference<LogData>() {
            });
            logData = desensitizationLog(logData);
            if (null == logData) {
                return;
            }
            collector.collect(logData);
        } catch (Exception e) {
            log.error("数据序列化失败:{}", value.getField(0).toString());
        }
    }

    private LogData desensitizationLog(LogData logData) {
        if (hasMatchHostname) {
            // 过滤主机
            String hostname = logData.getDimensions().get(GeneralConstants.HOSTNAME);
            if (null == hostname) {
                log.error("存在hostname为空的日志:{}", JSON.toJSONString(logData));
                return null;
            }
            if (!hostname.contains(jobConfig.getMatchHostname())) {
                log.debug("主机不包含{},数据{}丢弃;", hostname, JSON.toJSONString(logData));
                return null;
            }
        }
        //根据日志事件的timestamp做过滤
        Long timestamp = DateUtils.utc2timestamp(logData.getTimestamp());
        if (null == timestamp || timestamp.compareTo(jobConfig.getStartTimestamp()) < 0 ||
                timestamp.compareTo(jobConfig.getEndTimestamp()) > 0) {
            log.error("时间:{}不符合预期,数据丢弃", timestamp);
            return null;
        }
        // 正常数据脱敏
        Map<String, String> normalFieldsMap = logData.getNormalFields();
        try {
            // 1 脱敏普通列中key
            desensitizationNormalFieldByKey(normalFieldsMap);
            // 2 脱敏message
            desensitizationNormalFieldMessage(normalFieldsMap);
        } catch (Exception e) {
            log.error("数据：{};解析错误：异常信息：{}", JSON.toJSONString(logData), e);
        }
        return logData;
    }

    /**
     * 完成主体逻辑
     * //TODO 待精简逻辑
     *
     * @param normalFieldsMap 普通列
     */
    public void desensitizationNormalFieldMessage(Map<String, String> normalFieldsMap) {
        if (!normalFieldsMap.containsKey(GeneralConstants.MESSAGE)) {
            log.error("日志无message字段");
            return;
        }
        String sourceMessage = normalFieldsMap.get(GeneralConstants.MESSAGE);
        if (null == sourceMessage) {
            log.error("日志message为空");
            normalFieldsMap.put(GeneralConstants.ZORKDATA_FLAG, ZorkdataFlag.STR2);
            return;
        }
        // 脱敏逻辑
        if (sourceMessage.contains(GeneralConstants.ANS)) {
            if (!sourceMessage.contains(GeneralConstants.REQ)) {
                // 有Ans 没Req
                normalFieldsMap.put(GeneralConstants.ZORKDATA_FLAG, ZorkdataFlag.STR3);
                StringBuilder messageSuffixDesc = new StringBuilder();
                if (sourceMessage.contains(GeneralConstants.CODE_MSG)) {
                    String[] ansArrTemp = sourceMessage.split(GeneralConstants.CODE_MSG);
                    String ansArr = ansArrTemp[1];
                    // 此处还有其他情况
                    String[] ans = ansArr.split(GeneralConstants.SEMICOLON);
                    if (ans.length > GeneralConstants.NUM5) {
                        String[] ansTemp = new String[5];
                        ansTemp[0] = ans[0];
                        ansTemp[1] = ans[1];
                        ansTemp[2] = ans[2];
                        StringBuilder valueString = new StringBuilder();
                        for (int i = 3; i < ans.length - 1; i++) {
                            valueString.append(ans[i]).append(GeneralConstants.SEMICOLON);
                        }
                        ansTemp[3] = valueString.toString();
                        ansTemp[4] = ans[ans.length - 1];
                        ans = ansTemp;
                    }
                    String keysString = null;
                    String valuesString = null;
                    if (ans.length >= GeneralConstants.NUM4) {
                        keysString = ans[2];
                        valuesString = ans[3];
                    } else {
                        int commaIndex = ansArr.indexOf(GeneralConstants.SEMICOLON);
                        keysString = ansArr.substring(0, commaIndex);
                        valuesString = ansArr.substring(commaIndex);
                        if (ansArr.length() > commaIndex) {
                            valuesString = ansArr.substring(commaIndex + 1);
                        }
                    }
                    String[] keysArr = keysString.split(GeneralConstants.COMMA);
                    List<Integer> valueIndex = new ArrayList<>();
                    for (int i = 0; i < keysArr.length; i++) {
                        String key = keysArr[i].trim();
                        if (jobConfig.getDesensitizationKeyList().contains(key)) {
                            valueIndex.add(i);
                        }
                    }
                    if (ans.length >= GeneralConstants.NUM4) {
                        messageSuffixDesc
                                .append(ansArrTemp[0])
                                .append(GeneralConstants.CODE_MSG)
                                .append(ans[0]).append(GeneralConstants.SEMICOLON)
                                .append(ans[1]).append(GeneralConstants.SEMICOLON)
                                .append(ans[2]).append(GeneralConstants.SEMICOLON);
                    } else {
                        messageSuffixDesc
                                .append(ansArrTemp[0])
                                .append(GeneralConstants.CODE_MSG)
                                .append(ansArrTemp[1].split(GeneralConstants.SEMICOLON)[0])
                                .append(GeneralConstants.SEMICOLON);
                    }
                    String[] valueArr = valuesString.split(GeneralConstants.COMMA);
                    int valueSize = valueArr.length / keysArr.length + 1;
                    if ((valueArr.length + 1) % (keysArr.length) != 0) {
                        valueSize = valueArr.length / keysArr.length;
                    }
                    List<Integer> ansValueIndex = getAnsValueIndex(valueIndex, valueSize, keysArr.length);
                    for (int i = 0; i < valueArr.length; i++) {
                        String value = valueArr[i];
                        if (ansValueIndex.contains(i)) {
                            // 需要脱敏的字段
                            value = string2DesensitizationString(value);
                            valueArr[i] = value;
                        }
                        if (i == valueArr.length - 1) {
                            messageSuffixDesc.append(value);
                        } else {
                            messageSuffixDesc.append(value).append(GeneralConstants.COMMA);
                        }
                    }
                    if (ans.length >= GeneralConstants.NUM4) {
                        messageSuffixDesc.append(GeneralConstants.SEMICOLON);
                        for (int i = 4; i < ans.length; i++) {
                            messageSuffixDesc.append(ans[i]).append(GeneralConstants.SEMICOLON);
                        }
                    }
                } else {
                    log.error("message不包含CODE,MSG;");
                    messageSuffixDesc.append(messageSuffixDesc);
                }
                normalFieldsMap.put(GeneralConstants.MESSAGE, messageSuffixDesc.toString());
                return;
            }
            // 既包含Ans也包含Req
            String[] sourceMessageArr = sourceMessage.split(GeneralConstants.ANS);
            String messagePrefix = sourceMessageArr[0];
            String[] reqArr = messagePrefix.split(GeneralConstants.SYMBOL_STR);
            StringBuilder messagePrefixDesc = new StringBuilder();
            for (int i = 0; i < reqArr.length; i++) {
                if (reqArr[i].contains(GeneralConstants.EQUAL_SIGN)) {
                    String[] reqItemArr = reqArr[i].split(GeneralConstants.EQUAL_SIGN);
                    if (reqItemArr.length == 2) {
                        String key = reqItemArr[0];
                        String value = reqItemArr[1];
                        if (jobConfig.getDesensitizationKeyList().contains(key)) {
                            value = string2DesensitizationString(value);
                        }
                        messagePrefixDesc
                                .append(key)
                                .append(GeneralConstants.EQUAL_SIGN)
                                .append(value);
                    } else {
                        messagePrefixDesc.append(reqArr[i]);
                    }
                } else {
                    messagePrefixDesc.append(GeneralConstants.SYMBOL_STR);
                }
                if (i != reqArr.length - 1) {
                    messagePrefixDesc.append(GeneralConstants.SYMBOL_STR);
                }
            }

            String messageSuffix = sourceMessageArr[1];
            StringBuilder messageSuffixDesc = new StringBuilder();
            if (messageSuffix.contains(GeneralConstants.CODE_MSG)) {
                String[] ansArrTemp = messageSuffix.split(GeneralConstants.CODE_MSG);
                String ansArr = ansArrTemp[1];
                // 此处还有其他情况
                String[] ans = ansArr.split(GeneralConstants.SEMICOLON);
                if (ans.length > GeneralConstants.NUM5) {
                    String[] ansTemp = new String[5];
                    ansTemp[0] = ans[0];
                    ansTemp[1] = ans[1];
                    ansTemp[2] = ans[2];
                    StringBuilder valueString = new StringBuilder();
                    for (int i = 3; i < ans.length - 1; i++) {
                        valueString.append(ans[i]).append(GeneralConstants.SEMICOLON);
                    }
                    ansTemp[3] = valueString.toString();
                    ansTemp[4] = ans[ans.length - 1];
                    ans = ansTemp;
                }
                String keysString = null;
                String valuesString = null;
                if (ans.length >= GeneralConstants.NUM5) {
                    keysString = ans[2];
                    valuesString = ans[3];
                } else {
                    int commaIndex = ansArr.indexOf(GeneralConstants.SEMICOLON);
                    keysString = ansArr.substring(0, commaIndex);
                    valuesString = ansArr.substring(commaIndex);
                    if (ansArr.length() > commaIndex) {
                        valuesString = ansArr.substring(commaIndex + 1);
                    }
                }
                String[] keysArr = keysString.split(GeneralConstants.COMMA);
                List<Integer> valueIndex = new ArrayList<>();
                for (int i = 0; i < keysArr.length; i++) {
                    String key = keysArr[i].trim();
                    if (jobConfig.getDesensitizationKeyList().contains(key)) {
                        valueIndex.add(i);
                    }
                }
                if (ans.length >= GeneralConstants.NUM4) {
                    messageSuffixDesc
                            .append(ansArrTemp[0])
                            .append(GeneralConstants.CODE_MSG)
                            .append(ans[0]).append(GeneralConstants.SEMICOLON)
                            .append(ans[1]).append(GeneralConstants.SEMICOLON)
                            .append(ans[2]).append(GeneralConstants.SEMICOLON);
                } else {
                    messageSuffixDesc
                            .append(ansArrTemp[0])
                            .append(GeneralConstants.CODE_MSG)
                            .append(ansArrTemp[1].split(GeneralConstants.SEMICOLON)[0])
                            .append(GeneralConstants.SEMICOLON);
                }
                String[] valueArr = valuesString.split(GeneralConstants.COMMA);
                int valueSize = valueArr.length / keysArr.length + 1;
                if ((valueArr.length + 1) % (keysArr.length) != 0) {
                    valueSize = valueArr.length / keysArr.length;
                }
                List<Integer> ansValueIndex = getAnsValueIndex(valueIndex, valueSize, keysArr.length);
                for (int i = 0; i < valueArr.length; i++) {
                    String value = valueArr[i];
                    if (ansValueIndex.contains(i)) {
                        // 需要脱敏的字段
                        value = string2DesensitizationString(value);
                        valueArr[i] = value;
                    }
                    if (i == valueArr.length - 1) {
                        messageSuffixDesc.append(value);
                    } else {
                        messageSuffixDesc.append(value).append(GeneralConstants.COMMA);
                    }
                }
                if (ans.length >= GeneralConstants.NUM4) {
                    messageSuffixDesc.append(GeneralConstants.SEMICOLON);
                    for (int i = 4; i < ans.length; i++) {
                        messageSuffixDesc.append(ans[i]).append(GeneralConstants.SEMICOLON);
                    }
                }
            } else {
                log.error("message不包含CODE,MSG;");
                messageSuffixDesc.append(messageSuffixDesc);
            }
            normalFieldsMap.put(GeneralConstants.MESSAGE,
                    messagePrefixDesc + GeneralConstants.ANS + messageSuffixDesc);
        } else if (sourceMessage.contains(GeneralConstants.REQ)) {
            // 仅仅包含Req 不包含Ans
            String[] reqArr = sourceMessage.split(GeneralConstants.SYMBOL_STR);
            StringBuilder messagePrefixDesc = new StringBuilder();
            for (int i = 0; i < reqArr.length; i++) {
                if (reqArr[i].contains(GeneralConstants.EQUAL_SIGN)) {
                    String[] reqItemArr = reqArr[i].split(GeneralConstants.EQUAL_SIGN);
                    if (reqItemArr.length == 2) {
                        String key = reqItemArr[0];
                        String value = reqItemArr[1];
                        if (jobConfig.getDesensitizationKeyList().contains(key)) {
                            value = string2DesensitizationString(value);
                        }
                        messagePrefixDesc
                                .append(key)
                                .append(GeneralConstants.EQUAL_SIGN)
                                .append(value);
                    } else {
                        messagePrefixDesc.append(reqArr[i]);
                    }
                } else {
                    messagePrefixDesc.append(GeneralConstants.SYMBOL_STR);
                }
                if (i != reqArr.length - 1) {
                    messagePrefixDesc.append(GeneralConstants.SYMBOL_STR);
                }
            }
            normalFieldsMap.put(GeneralConstants.MESSAGE, messagePrefixDesc.toString());
        } else {
            // 不包含Ans 不包含Req
            log.error("数据既不存在Req也不存在Ans部分");
            normalFieldsMap.put(GeneralConstants.ZORKDATA_FLAG, ZorkdataFlag.STR1);
        }
    }

    /**
     * 获取ans部分需要脱敏的下标
     *
     * @param keyIndex  需要脱敏的key下标
     * @param valueSize value的步长
     * @param keySize   key的步长
     * @return 需要脱敏的下标
     */
    private List<Integer> getAnsValueIndex(List<Integer> keyIndex, int valueSize, int keySize) {
        List<Integer> valueIndex = new ArrayList<>();
        for (Integer index : keyIndex) {
            for (int i = 0; i < valueSize; i++) {
                if ((index + keySize * i) > keySize) {
                    valueIndex.add(index + keySize * i - 1);
                } else {
                    valueIndex.add(index + keySize * i);
                }
            }
        }
        return valueIndex;
    }

    /**
     * 脱敏普通列
     *
     * @param normalFieldsMap 普通列map
     */
    private void desensitizationNormalFieldByKey(Map<String, String> normalFieldsMap) {
        if (null == normalFieldsMap) {
            log.error("普通列为空，不需要脱敏;");
        }
        Set<Map.Entry<String, String>> entrySet = normalFieldsMap.entrySet();
        Iterator<Map.Entry<String, String>> iterator = entrySet.iterator();
        while (iterator.hasNext()) {
            Map.Entry<String, String> next = iterator.next();
            String key = next.getKey();
            // message 或不包含在需要脱敏名单中的往下继续执行
            if (GeneralConstants.MESSAGE.equals(key)
                    || !jobConfig.getDesensitizationKeyList().contains(key)) {
                continue;
            }
            String value = next.getValue();
            String targetValue = string2DesensitizationString(value);
            next.setValue(targetValue);
        }
    }

    /**
     * 普通字符串脱敏
     *
     * @param sourceString 原始字符串
     * @return 脱敏后字符串
     */
    public String string2DesensitizationString(String sourceString) {
        if (null == sourceString) {
            return null;
        }
        int length = sourceString.length();
        return chang2String(length);
    }

    /**
     * 根据字符串长度脱敏成‘*’StringBuilder
     *
     * @param length 字符串长度
     * @return 脱敏后 ** 字符串
     */
    private StringBuilder chang2StringBuilder(int length) {
        StringBuilder result = new StringBuilder();
        for (int i = 0; i < length; i++) {
            result.append("*");
        }
        return result;
    }

    /**
     * 根据字符串长度脱敏成‘*’字符串
     *
     * @param length 字符串长度
     * @return 脱敏后 ** 字符串
     */
    private String chang2String(int length) {
        return chang2StringBuilder(length).toString();
    }


}
