package com.zorkdata.desensitization.utils;

import com.alibaba.fastjson.JSON;
import com.zorkdata.desensitization.constans.GeneralConstants;
import com.zorkdata.desensitization.exception.ZorkException;
import lombok.extern.slf4j.Slf4j;
import org.apache.flink.api.java.utils.ParameterTool;

import java.io.*;
import java.util.*;

/**
 * @author: LiaoMingtao
 * @date: 2020/8/7
 */
@Slf4j
public class PropertiesUtil {

    private static final int DEFAULT_PARAMS_MAP_LENGTH = 10;
    private static final String REGULAR = "regular";


    public static void main(String[] args) {
        List<String> propertiesContentList = PropertiesUtil.getPropertiesContentList("/regular");
        System.out.println(JSON.toJSONString(propertiesContentList));
        Map<String, String> propertiesMap = getPropertiesMap(propertiesContentList);
        System.out.println(JSON.toJSONString(propertiesMap));
    }


    /**
     * 获取配置文件map
     *
     * @param args 入参参数
     * @return 配置文件map
     */
    public static Map<String, String> getPropertiesMap(String[] args) throws ZorkException {
        String configPath = null;
        ParameterTool parameterTool = ParameterTool.fromArgs(args);
        configPath = parameterTool.get(REGULAR);
        log.info("read config path is {}", configPath);
        List<String> propertiesContentList = PropertiesUtil.getPropertiesContentList(configPath);
        Map<String, String> confMap = PropertiesUtil.getPropertiesMap(propertiesContentList);
        if (confMap.isEmpty()) {
            log.error("配置文件regular不存在,系统退出");
            throw new ZorkException("配置文件regular不存在,系统退出");
        }
        return confMap;
    }

    /**
     * 通过配置文件每行字符串获取文件所有内容
     *
     * @param contentList 文件内容
     * @return 键值对map
     */
    public static Map<String, String> getPropertiesMap(List<String> contentList) {
        Map<String, String> map = new HashMap<>(GeneralConstants.MAXIMUM_CAPACITY);
        if (null == contentList) {
            return map;
        }
        for (String item : contentList) {
            if (item.contains(GeneralConstants.EQUAL_SIGN)) {
                int start = item.indexOf(GeneralConstants.EQUAL_SIGN);
                String key = item.substring(0, start);
                String value = GeneralConstants.EMPTY_STR;
                if (item.length() > start) {
                    value = item.substring(start + 1);
                }
                map.put(key, value);
            }
        }
        return map;
    }

    /**
     * 按照行读取文件
     *
     * @param propertiesFileName 文件路径
     * @return list，每行为一项
     */
    public static List<String> getPropertiesContentList(String propertiesFileName) {
        InputStream inputStream = null;
        List<String> resultList = new ArrayList<>();
        try {
            inputStream = new FileInputStream(propertiesFileName);
            //存放读的字节，就是读的结果
            BufferedReader bufferedReader = new BufferedReader(new InputStreamReader(inputStream));
            String str = null;
            while ((str = bufferedReader.readLine()) != null) {
                resultList.add(str);
            }
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        return resultList;
    }

    /**
     * 读取配置文件的所有内容
     *
     * @param propertiesFileName 配置文件名称
     * @return 配置文件所有内容
     */
    public static String getPropertiesContent(String propertiesFileName) {
        InputStream inputStream = null;
        StringBuilder stringBuilder = new StringBuilder();
        try {
            inputStream = PropertiesUtil.class.getResourceAsStream(propertiesFileName);
            //存放读的字节，就是读的结果
            int result = -1;
            while ((result = inputStream.read()) != -1) {
                stringBuilder.append((char) result);
            }
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        return stringBuilder.toString();
    }

    /**
     * 根据文件名获取该properties对象
     *
     * @param propertiesFileName 配置文件名称
     * @return properties
     */
    public static Properties getProperties(String propertiesFileName) {
        Properties properties = new Properties();
        InputStream inputStream = null;
        try {
            inputStream = PropertiesUtil.class.getResourceAsStream(propertiesFileName);
            properties.load(inputStream);
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        return properties;
    }
}
