package com.zorkdata.desensitization.utils;

import lombok.extern.slf4j.Slf4j;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * @author: LiaoMingtao
 * @date: 2020/10/22
 */
@Slf4j
public class DateUtil {

    private static final String NULL = "";
    private static final String BAR_STRING = "-";
    private static final String TIME_ZONE = "Asia/shanghai";
    private static ThreadLocal<SimpleDateFormat> UTC_FORMATTER = ThreadLocal.withInitial(() -> new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS+08:00"));
    private static ThreadLocal<SimpleDateFormat> YYYY_MM_DD_HH_MM_SS_FORMATTER = ThreadLocal.withInitial(() -> new SimpleDateFormat("yyyy-MM-dd HH:mm:ss"));
    private static ThreadLocal<SimpleDateFormat> YYYY_MM_DD_FORMATTER = ThreadLocal.withInitial(() -> new SimpleDateFormat("yyyy-MM-dd"));


    /**
     * 形如yyyy-MM-dd转yyyymmdd
     *
     * @param time 日期
     * @return yyyymmdd的日期
     */
    public static String date2date(String time) {
        return time.replace(BAR_STRING, NULL);
    }

    /**
     * 批量转日期 去除日期中的‘-’
     *
     * @param timeList 例如 2019-10-11 的日期列表
     * @return List<String>
     */
    public static List<String> date2date(List<String> timeList) {
        List<String> resultList = new ArrayList<>();
        for (String time : timeList) {
            String tempTime = date2date(time);
            resultList.add(tempTime);
        }
        return resultList;
    }


    /**
     * 获取两个时间段内的所有日期，日期可跨年
     *
     * @param begin 开始时间 yyyy-MM-dd HH:mm:ss
     * @param end   结束时间 yyyy-MM-dd HH:mm:ss
     * @return List<String>
     */
    public static List<String> getBetweenDate(String begin, String end) {
        List<String> betweenList = new ArrayList<>();
        Calendar startDay = Calendar.getInstance();
        try {
            startDay.setTime(YYYY_MM_DD_HH_MM_SS_FORMATTER.get().parse(begin));
            startDay.add(Calendar.DATE, -1);

            while (true) {
                startDay.add(Calendar.DATE, 1);
                Date newDate = startDay.getTime();
                String newEnd = YYYY_MM_DD_FORMATTER.get().format(newDate);
                betweenList.add(newEnd);
                if (end.startsWith(newEnd)) {
                    break;
                }
            }
        } catch (ParseException e) {
            log.error(String.valueOf(e));
        }
        return betweenList;
    }

    /**
     * 普通时间转时间戳
     *
     * @param time 普通时间
     * @return 时间戳
     */
    public static long time2Timestamp(String time) {
        Date gpsUtcDate = null;
        try {
            gpsUtcDate = YYYY_MM_DD_HH_MM_SS_FORMATTER.get().parse(time);
            return gpsUtcDate.getTime();
        } catch (ParseException e) {
            log.error("时间戳格式转换异常：{} 原因: {}", time, e.getMessage());
        }
        return 0L;
    }

    /**
     * UTC时间转
     *
     * @param utcTime UTC时间
     * @return unix时间戳
     */
    public static Long utc2timestamp(String utcTime) {
        //时区定义并进行时间获取
        // UTC_FORMATTER.get().setTimeZone(TimeZone.getTimeZone(TIME_ZONE));
        Date gpsUtcDate = null;
        try {
            gpsUtcDate = UTC_FORMATTER.get().parse(utcTime);
        } catch (ParseException e) {
            log.error("时间戳格式转换异常：{} 原因: {}", utcTime, e.getMessage());
            return null;
        }
        return gpsUtcDate.getTime();
    }

}
