package com.zorkdata.tools.kafka;

import com.zorkdata.tools.oldkafka.AvroSerializerFactory;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.ProducerConfig;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.common.serialization.StringSerializer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Map;
import java.util.Properties;


/**
 * @author MSI-Gaming
 */
public class CommonProducer {

    private static final Logger logger = LoggerFactory.getLogger(CommonProducer.class);

    private String kafkaServer;
    private int kafkaBathSize;

    private KafkaProducer<String, byte[]> producerByte;
    private KafkaProducer<String, String> producerString;

    public void initConfig() {
        kafkaServer = Config.INSTANCE.getKafkaServers();
        kafkaBathSize = Config.INSTANCE.getKafkaBathSize();
    }

    public CommonProducer() {
        try {
            initConfig();
            Properties props = new Properties();
            props.put("bootstrap.servers", kafkaServer);
            // props.put("client.id", "webAPI4LogGather"); 不自定义client.id,使用自增
            props.put("key.serializer", "org.apache.kafka.common.serialization.StringSerializer");
            props.put("value.serializer", "org.apache.kafka.common.serialization.ByteArraySerializer");
            props.put("batch.size", kafkaBathSize);
            // 启用压缩
            props.put("compression.type", "lz4");
            producerByte = new KafkaProducer<>(props);
            props.put("value.serializer", "org.apache.kafka.common.serialization.StringSerializer");
            producerString = new KafkaProducer<>(props);
        } catch (Exception ex) {
            logger.error(ex.toString());
        }
    }

    /**
     * 发送logAvro数据
     *
     * @param topic        指标名称
     * @param logTypeName  日志结构名称
     * @param timestamp    时间
     * @param source       日志路径
     * @param offset       偏移量
     * @param dimensions   维度
     * @param metrics      指标
     * @param normalFields 普通列
     */
    public void sendLogAvro(String topic, String logTypeName, String timestamp, String source, String offset,
                            Map<String, String> dimensions, Map<String, Double> metrics, Map<String, String> normalFields) {
        try {
            byte[] bytes = AvroSerializerFactory.getLogAvorSerializer().serializingLog(logTypeName, timestamp, source,
                    offset, dimensions, metrics, normalFields);
            producerByte.send(new ProducerRecord<>(topic, null, bytes));
        } catch (Exception e) {
            logger.error(e.toString());
        }
    }

    /**
     * 想kafka发送metricAvro数据
     *
     * @param metricTopic   topic名称
     * @param metricSetName 指标集名称
     * @param timestamp     时间戳
     * @param dimensions    维度
     * @param metrics       指标
     */
    public void sendMetricAvro(String metricTopic, String metricSetName, String timestamp, Map<String, String> dimensions,
                               Map<String, Double> metrics) {
        try {
            byte[] bytes = AvroSerializerFactory.getMetricAvorSerializer().serializingMetric(metricSetName, timestamp,
                    dimensions, metrics);
            producerByte.send(new ProducerRecord<>(metricTopic, null, bytes));
        } catch (Exception e) {
            logger.error(e.toString());
        }
    }

    /**
     * 向kafka发送json数据
     *
     * @param topic   topic名称
     * @param logJson json数据
     */
    public void sendJson(String topic, String logJson) {
        try {
            producerString.send(new ProducerRecord<>(topic, null, logJson));
        } catch (Exception e) {
            logger.error(e.toString());
        }
    }


}
